/* 
 * Copyright (C) 2004, 2005 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <eel/eel.h>
#include "gt-shell.h"
#include "gt-conf.h"
#include "gt-util.h"

#define CONF_HTTP_PROXY_NAMESPACE		"/system/http_proxy"
#define CONF_HTTP_PROXY_USE_HTTP_PROXY		CONF_HTTP_PROXY_NAMESPACE "/use_http_proxy"
#define CONF_HTTP_PROXY_HOST			CONF_HTTP_PROXY_NAMESPACE "/host"
#define CONF_HTTP_PROXY_PORT			CONF_HTTP_PROXY_NAMESPACE "/port"
#define CONF_HTTP_PROXY_USE_AUTHENTICATION	CONF_HTTP_PROXY_NAMESPACE "/use_authentication"
#define CONF_HTTP_PROXY_AUTHENTICATION_USER	CONF_HTTP_PROXY_NAMESPACE "/authentication_user"
#define CONF_HTTP_PROXY_AUTHENTICATION_PASSWORD	CONF_HTTP_PROXY_NAMESPACE "/authentication_password"
#define CONF_PROXY_NAMESPACE			"/system/proxy"
#define CONF_PROXY_MODE				CONF_PROXY_NAMESPACE "/mode"
#define CONF_PROXY_SOCKS_HOST			CONF_PROXY_NAMESPACE "/socks_host"
#define CONF_PROXY_SOCKS_PORT			CONF_PROXY_NAMESPACE "/socks_port"

struct _GTShellPrivate
{
  TranslateSession	*translate_session;
};

GTShell *gt_shell = NULL;

static void gt_shell_class_init (GTShellClass *class);
static void gt_shell_init (GTShell *shell);

static void gt_shell_proxy_notify_cb (GConfClient *client,
				      unsigned int cnxn_id,
				      GConfEntry *entry,
				      gpointer user_data);
static void gt_shell_services_notify_cb (GConfClient *client,
					 unsigned int cnxn_id,
					 GConfEntry *entry,
					 gpointer user_data);

static void gt_shell_set_proxy (GTShell *shell);
static void gt_shell_set_services (GTShell *shell);

GType
gt_shell_get_type (void)
{
  static GType type = 0;
  
  if (! type)
    {
      static const GTypeInfo info = {
	sizeof(GTShellClass),
	NULL,
	NULL,
	(GClassInitFunc) gt_shell_class_init,
	NULL,
	NULL,
	sizeof(GTShell),
	0,
	(GInstanceInitFunc) gt_shell_init,
      };
      
      type = g_type_register_static(G_TYPE_OBJECT, "GTShell", &info, 0);
    }

  return type;
}

static void
gt_shell_class_init (GTShellClass *class)
{
  g_type_class_add_private(class, sizeof(GTShellPrivate));
}

static void
gt_shell_init (GTShell *shell)
{
  g_return_if_fail(gt_shell == NULL);
  gt_shell = shell;
  eel_add_weak_pointer(&gt_shell);

  shell->priv = G_TYPE_INSTANCE_GET_PRIVATE(shell, GT_TYPE_SHELL, GTShellPrivate);
  shell->priv->translate_session = translate_session_new(NULL);

  gt_shell_set_proxy(shell);
  gt_shell_set_services(shell);

  eel_gconf_notification_add(GT_CONF_SERVICES, gt_shell_services_notify_cb, shell);

  eel_gconf_monitor_add(CONF_HTTP_PROXY_NAMESPACE);
  eel_gconf_notification_add(CONF_HTTP_PROXY_NAMESPACE, gt_shell_proxy_notify_cb, shell);

  eel_gconf_monitor_add(CONF_PROXY_NAMESPACE);
  eel_gconf_notification_add(CONF_PROXY_NAMESPACE, gt_shell_proxy_notify_cb, shell);
}

TranslateSession *
gt_shell_get_translate_session (GTShell *shell)
{
  g_return_val_if_fail(GT_IS_SHELL(shell), NULL);

  return shell->priv->translate_session;
}

static void
gt_shell_proxy_notify_cb (GConfClient *client,
			  unsigned int cnxn_id,
			  GConfEntry *entry,
			  gpointer user_data)
{
  GTShell *shell = user_data;

  gt_shell_set_proxy(shell);
}

static void
gt_shell_services_notify_cb (GConfClient *client,
			     unsigned int cnxn_id,
			     GConfEntry *entry,
			     gpointer user_data)
{
  GTShell *shell = user_data;

  gt_shell_set_services(shell);
}

static void
gt_shell_set_proxy (GTShell *shell)
{
  char *uri = NULL;
  char *mode;

  g_return_if_fail(GT_IS_SHELL(shell));

  mode = eel_gconf_get_string(CONF_PROXY_MODE);
  if (mode)
    {
      if (! strcmp(mode, "manual"))
	{
	  char *socks_host;

	  socks_host = eel_gconf_get_string(CONF_PROXY_SOCKS_HOST);
	  if (socks_host)
	    {
	      if (*socks_host)
		{
		  int socks_port;

		  socks_port = eel_gconf_get_integer(CONF_PROXY_SOCKS_PORT);
		  if (socks_port <= 0 || socks_port > 65535)
		    socks_port = 1080;

		  /* we assume it's a Socks 5 proxy, since it is not specified */
		  uri = g_strdup_printf("socks5://%s:%i", socks_host, socks_port);
		}

	      g_free(socks_host);
	    }
	}
      g_free(mode);
    }
  
  if (! uri)
    {
      if (eel_gconf_get_boolean(CONF_HTTP_PROXY_USE_HTTP_PROXY))
	{
	  char *host;

	  /* FIXME: also honour /system/http_proxy/ignore_hosts */

	  host = eel_gconf_get_string(CONF_HTTP_PROXY_HOST);
	  if (host)
	    {
	      if (*host)
		{
		  GString *string;
		  int port;
	      
		  string = g_string_new("http://");
		  
		  port = eel_gconf_get_integer(CONF_HTTP_PROXY_PORT);
		  if (port <= 0 || port > 65535)
		    port = 8080; /* default as recommended by http-method.c of GnomeVFS */
		  
		  if (eel_gconf_get_boolean(CONF_HTTP_PROXY_USE_AUTHENTICATION))
		    {
		      char *authentication_user;
		      char *authentication_password;
		      
		      authentication_user = eel_gconf_get_string(CONF_HTTP_PROXY_AUTHENTICATION_USER);
		      authentication_password = eel_gconf_get_string(CONF_HTTP_PROXY_AUTHENTICATION_PASSWORD);
		      
		      if (authentication_user && authentication_password
			  && *authentication_user && *authentication_password)
			g_string_append_printf(string, "%s:%s@", authentication_user, authentication_password);
		      
		      g_free(authentication_user);
		      g_free(authentication_password);
		    }
	      
		  g_string_append_printf(string, "%s:%i", host, port);
		  uri = g_string_free(string, FALSE);
		}

	      g_free(host);
	    }
	}
    }

  translate_set_proxy(uri);
  g_free(uri);
}

static void
gt_shell_set_services (GTShell *shell)
{
  GSList *conf_services;
  GSList *services = NULL;
  GSList *all_services;
  GSList *l;

  g_return_if_fail(GT_IS_SHELL(shell));

  conf_services = gt_conf_get_services();
  GT_LIST_FOREACH(l, conf_services)
    {
      const GTConfService *conf_service = l->data;
      if (conf_service->enabled)
	services = g_slist_append(services, g_object_ref(conf_service->service));
    }

  all_services = translate_get_services();
  GT_LIST_FOREACH(l, all_services)
    {
      TranslateService *service = l->data;

      if (! gt_conf_services_get_from_name(conf_services, translate_service_get_name(service)))
	services = g_slist_append(services, g_object_ref(service));
    }

  gt_conf_services_free(conf_services);
  gt_g_object_slist_free(all_services);

  translate_session_set_services(shell->priv->translate_session, services);
  gt_g_object_slist_free(services);
}

GTShell *
gt_shell_new (void)
{
  return g_object_new(GT_TYPE_SHELL, NULL);
}
