/*
 * Copyright © 2013 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/**
 * \file bgra-errors.c
 *
 * This tests a few error conditions from the ARB_vertex_array_bgra extension.
 */

#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN
	config.supports_gl_compat_version = 10;
	config.window_visual = PIGLIT_GL_VISUAL_RGB | PIGLIT_GL_VISUAL_DOUBLE;
	config.khr_no_error_support = PIGLIT_HAS_ERRORS;
PIGLIT_GL_TEST_CONFIG_END

void
piglit_init(int argc, char **argv)
{
	static GLubyte ubytes[4] = { 255, 0, 0, 127 };
	static const GLenum invalid_types[] = {
		GL_BYTE,
		GL_SHORT,
		GL_UNSIGNED_SHORT,
		GL_INT,
		GL_UNSIGNED_INT,
		GL_FLOAT,
		GL_DOUBLE,
	};
	unsigned i;

	bool pass = true;

	piglit_require_gl_version(20);
	piglit_require_extension("GL_ARB_vertex_array_bgra");

	/* Test when size == GL_BGRA && normalized == GL_TRUE;
	 * should generate GL_NO_ERROR
	 */
	glVertexAttribPointer(0, GL_BGRA, GL_UNSIGNED_BYTE, GL_TRUE,
			      4 * sizeof(GLubyte), ubytes);
	pass = piglit_check_gl_error(GL_NO_ERROR) && pass;

	/* From the GL_ARB_vertex_array_bgra specification:
	 * "The error INVALID_VALUE is generated by VertexAttribPointer if
	 *  size is BGRA and normalized is FALSE."
	 */
	glVertexAttribPointer(1, GL_BGRA, GL_UNSIGNED_BYTE, GL_FALSE,
			      4 * sizeof(GLubyte), ubytes);
	pass = piglit_check_gl_error(GL_INVALID_OPERATION) && pass;

	/* From the GL_ARB_vertex_array_bgra specification:
	 * "The error INVALID_VALUE is generated when ColorPointer,
	 *  SecondaryPointer, or VertexAttribPointer is called with size
	 *  set to BGRA and type is not UNSIGNED_BYTE."
	 */
	for (i = 0; i < ARRAY_SIZE(invalid_types); i++) {
		glColorPointer(GL_BGRA, invalid_types[i], 0, NULL);
		pass = piglit_check_gl_error(GL_INVALID_OPERATION) && pass;

		glSecondaryColorPointer(GL_BGRA, invalid_types[i], 0, NULL);
		pass = piglit_check_gl_error(GL_INVALID_OPERATION) && pass;

		glVertexAttribPointer(1, GL_BGRA, invalid_types[i], GL_TRUE,
				      0, NULL);
		pass = piglit_check_gl_error(GL_INVALID_OPERATION) && pass;
	}

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}

enum piglit_result
piglit_display(void)
{
	/* Should never get here. */
	return PIGLIT_FAIL;
}
