//=======================================================================
// log.cc
//-----------------------------------------------------------------------
// This file is part of the package paco
// Copyright (C) 2004-2009 David Rosal
// For more information visit http://paco.sourceforge.net
//=======================================================================

#include "config.h"
#include "global.h"
#include "pkgset.h"
#include "pkg.h"
#include "log.h"
#include "info.h"
#include <fstream>
#include <sstream>
#include <iostream>
#include <iterator>
#include <algorithm>
#include <glob.h>

using namespace Paco;
using namespace std;

// Forward declarations
static string searchLibpaco();
static void setEnv(char const* var, string const& val);
static string stripSuffix(string const&);
static bool isNotValidPath(string const&);


Log::Log(Options const& opt)
:
	mOpt(opt),
	mAppend(opt.append()),
	mPkgName(opt.logPkg()),
	mTmpFile(),
	mFiles()
{
	if (opt.args().empty())
		getFilesFromStream(cin);
	else
		getFilesFromCommand();

	if (mFiles.empty())
		return;

	filterFiles();

	if (mPkgName.empty())
		writeFilesToStream(cout);
	else {
		if (!mFiles.empty())
			writeFilesToPkg();
		if (mAppend)
			gOut.dbgTitle("");
	}
}


Log::~Log()
{
	if (!mTmpFile.empty())
		unlink(mTmpFile.c_str());
}


// [static]
void Log::run(Options const& opt)
{
	static Log log(opt);
}


void Log::writeFilesToStream(ostream& s)
{
	copy(mFiles.begin(), mFiles.end(), ostream_iterator<string>(s, "\n"));
}


void Log::getFilesFromStream(istream& f)
{
	vector<string> x;
//	f.unsetf(ios::skipws);	// catch file names with whitespaces
	remove_copy_if(istream_iterator<string>(f), istream_iterator<string>(),
		back_inserter(x), isNotValidPath);
	transform(x.begin(), x.end(), inserter(mFiles, mFiles.begin()), realDir);
}


void Log::getFilesFromCommand()
{
	getTmpFile();

	pid_t pid = fork();

	if (pid == 0) { // child

		string command;
		string libpaco = searchLibpaco();
		
		for (unsigned i = 0; i < mOpt.args().size(); ++i)
			command += mOpt.args()[i] + " ";
		
		setEnv("PACO_TMPFILE", mTmpFile);
		setEnv("LD_PRELOAD", libpaco);
		setEnv("PACO_DEBUG", gOut.verbosity() > Out::VERBOSE ? "yes" : "");

		gOut.dbgTitle("settings");
		gOut.dbg("TMPFILE=" + mTmpFile + "\n"); 
		gOut.dbg("INCLUDE=" + mOpt.include() + "\n"); 
		gOut.dbg("EXCLUDE=" + mOpt.exclude() + "\n"); 
		gOut.dbg("IGNORE_ERRORS=" + string(mOpt.logIgnoreErrors() ? "1" : "0") + "\n");
		gOut.dbg("LD_PRELOAD=" + libpaco + "\n"); 
		gOut.dbg("command: " + command + "\n");
		gOut.dbgTitle("libpaco-log");

		char* cmd[] = { (char*)"sh", (char*)"-c", const_cast<char*>(command.c_str()), NULL };
		execv("/bin/sh", cmd);

		throw XErrno("execv()");
	}
	else if (pid == -1)
		throw XErrno("fork()");

	int status;
	waitpid(pid, &status, 0);
	gExitStatus = WIFEXITED(status) ? WEXITSTATUS(status) : EXIT_FAILURE_EXTERNAL;
	if (!mOpt.logIgnoreErrors() && gExitStatus != EXIT_SUCCESS)
		exit(gExitStatus);
	
	FileStream<ifstream> f(mTmpFile);
	getFilesFromStream(f);
}


void Log::getTmpFile()
{
	char* tmpFile = getenv("PACO_TMPFILE");
	if (tmpFile) {
		mTmpFile = tmpFile;
		return;
	}

	char* tmpDir = getenv("TMPDIR");
	char name[4096];
	snprintf(name, sizeof(name), "%s/pacoXXXXXX", tmpDir ? tmpDir : "/tmp");
	
	int fd = mkstemp(name);
	if (fd > 0) {
		fchmod(fd, 0644);
		close(fd);
	}
	else
		snprintf(name, sizeof(name), "/tmp/paco%d", getpid());

	mTmpFile = name;
}


void Log::removeAlreadyLoggedFiles()
{
	Pkg oldPkg(mPkgName);
	oldPkg.getFiles();

	vector<string> x;

	for (set<string>::iterator f = mFiles.begin(); f != mFiles.end(); ++f) {
		if (!oldPkg.hasFile(*f))
			x.push_back(*f);
	}
	mFiles.clear();
	copy(x.begin(), x.end(), inserter(mFiles, mFiles.begin()));
}


void Log::filterFiles()
{
	Out::Silencer* s = mPkgName.empty() ? new Out::Silencer() : NULL;

	vector<string> x;
	
	remove_copy_if(mFiles.begin(), mFiles.end(), back_inserter(x), Excluder(*this));

	mFiles.clear();
	
	if (x.empty()) {
		delete s;
		return;
	}

	gOut.dbgTitle("logged files");

	if (mPkgName.empty())
		copy(x.begin(), x.end(), inserter(mFiles, mFiles.begin()));
	else {
		// When writing the files to a log, strip suffixes (.bz2, .gz, ...)
		transform(x.begin(), x.end(), inserter(mFiles, mFiles.begin()), stripSuffix);
	}

	delete s;
}


void Log::writeFilesToLog(string const& logFile)
{
	ofstream f(logFile.c_str(), ios::app);
	if (!f)
		throw XErrno(logFile);
	copy(mFiles.begin(), mFiles.end(), ostream_iterator<string>(f, "\n"));
}


void Log::writeFilesToPkg()
{
	string const logFile = Config::logdir() + "/" + mPkgName;

	if (mAppend) {
		try {
			removeAlreadyLoggedFiles();
		}
		catch (...)	{
			mAppend = false;
		}
	}

	if (!mAppend) {
		mkdir(Config::logdir().c_str(), 0755);
		Info info(*this);
	}

	writeFilesToLog(logFile);

	Pkg::updateLog(logFile);

	if (gOut.verbosity() == Out::VERBOSE)
		writeFilesToStream(cerr);
}


//---------------//
// Log::Excluder //
//---------------//


Log::Excluder::Excluder(Log const& log)
:
	mInclude(log.mOpt.include()),
	mExclude(log.mOpt.exclude()),
	mLogMissing(log.mOpt.logMissing())
{ }


bool Log::Excluder::operator()(string const& path)
{
	if (inPaths(path, mExclude) || !inPaths(path, mInclude))
		return true;

	// non-existent files
	struct stat s;
	if (lstat(path.c_str(), &s) < 0)
		return !mLogMissing;

	// skip directories
	return S_ISDIR(s.st_mode);
}


//-------------------//
// Static free funcs //
//-------------------//


static bool isNotValidPath(string const& path)
{
	return UNLIKELY(path[0] == '#');
}


static string stripSuffix(string const& mPath)
{
	string path(mPath);
	string::size_type p;
	if ((p = path.rfind(".bz2")) == path.size() - 4
	||	(p = path.rfind(".gz")) == path.size() - 3)
		path.erase(p);
	gOut.dbg(path + "\n");
	return path;
}


//
// Search for libpaco-log.so in the filesystem.
// Take into account libpaco-log.so.0.1.0, libpaco-log.so.0.0 and so.
//
static string searchLibpaco()
{
	string libpath(LIBDIR "/libpaco-log.so");
	struct stat s;
	
	if (!stat(libpath.c_str(), &s))
		return libpath;
	
	glob_t g;
	memset(&g, 0, sizeof(g));
	
	if (!glob(LIBDIR "/libpaco-log.so.[0-9]*", GLOB_NOSORT,0, &g) && g.gl_pathc)
		libpath = g.gl_pathv[0];
	
	globfree(&g);
	
	return libpath;
}


static void setEnv(char const* var, string const& val)
{
#if HAVE_SETENV
	if (setenv(var, val.c_str(), 1) < 0)
		throw XErrno(string("setenv(") + var + ", " + val + ", 1)");
#else
	string str(var + "=" + val);
	if (putenv(str.c_str()) < 0)
		throw XErrno("putenv(" + str + ")");
#endif
}


