#############################################################################
##
## Copyright (C) 2018 The Qt Company Ltd.
## Contact: https://www.qt.io/licensing/
##
## This file is part of Qt for Python.
##
## $QT_BEGIN_LICENSE:LGPL$
## Commercial License Usage
## Licensees holding valid commercial Qt licenses may use this file in
## accordance with the commercial license agreement provided with the
## Software or, alternatively, in accordance with the terms contained in
## a written agreement between you and The Qt Company. For licensing terms
## and conditions see https://www.qt.io/terms-conditions. For further
## information use the contact form at https://www.qt.io/contact-us.
##
## GNU Lesser General Public License Usage
## Alternatively, this file may be used under the terms of the GNU Lesser
## General Public License version 3 as published by the Free Software
## Foundation and appearing in the file LICENSE.LGPL3 included in the
## packaging of this file. Please review the following information to
## ensure the GNU Lesser General Public License version 3 requirements
## will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
##
## GNU General Public License Usage
## Alternatively, this file may be used under the terms of the GNU
## General Public License version 2.0 or (at your option) the GNU General
## Public license version 3 or any later version approved by the KDE Free
## Qt Foundation. The licenses are as published by the Free Software
## Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
## included in the packaging of this file. Please review the following
## information to ensure the GNU General Public License requirements will
## be met: https://www.gnu.org/licenses/gpl-2.0.html and
## https://www.gnu.org/licenses/gpl-3.0.html.
##
## $QT_END_LICENSE$
##
#############################################################################
from build_scripts.options import has_option
from build_scripts.options import option_value
from build_scripts.utils import install_pip_dependencies, expand_clang_variables
from build_scripts.utils import get_qtci_virtualEnv
from build_scripts.utils import run_instruction
from build_scripts.utils import rmtree
from build_scripts.utils import get_python_dict
from build_scripts.utils import get_ci_qmake_path
from build_scripts.utils import provisioning
import os
import datetime
import calendar
import site
import sys

# Values must match COIN thrift
CI_HOST_OS = option_value("os")
CI_TARGET_OS = option_value("targetOs")
CI_HOST_ARCH = option_value("hostArch")
CI_TARGET_ARCH = option_value("targetArch")
CI_HOST_OS_VER = option_value("osVer")
CI_ENV_INSTALL_DIR = option_value("instdir")
CI_ENV_AGENT_DIR = option_value("agentdir")
CI_COMPILER = option_value("compiler")
CI_INTEGRATION_ID = option_value("coinIntegrationId") or str(calendar.timegm(datetime.datetime.now().timetuple()))
CI_FEATURES = []
_ci_features = option_value("features")
if _ci_features is not None:
    for f in _ci_features.split(', '):
        CI_FEATURES.append(f)
CI_RELEASE_CONF = has_option("packaging")
CI_TEST_PHASE = option_value("phase")
if CI_TEST_PHASE not in ["ALL", "BUILD", "WHEEL"]:
    CI_TEST_PHASE = "ALL"



def get_current_script_path():
    """ Returns the absolute path containing this script. """
    try:
        this_file = __file__
    except NameError:
        this_file = sys.argv[0]
    this_file = os.path.abspath(this_file)
    return os.path.dirname(this_file)

def is_snapshot_build():
    """
    Returns True if project needs to be built with --snapshot-build

    This is true if the version found in pyside_version.py is not a
    pre-release version (no alphas, betas).

    This eliminates the need to remove the --snapshot-build option
    on a per-release branch basis (less things to remember to do
    for a release).
    """
    setup_script_dir = get_current_script_path()
    pyside_version_py = os.path.join(
        setup_script_dir, "sources", "pyside2", "pyside_version.py")
    d = get_python_dict(pyside_version_py)

    release_version_type = d['release_version_type']
    pre_release_version = d['pre_release_version']
    if pre_release_version and release_version_type:
        return True
    return False

def call_setup(python_ver, phase):
    print("call_setup")
    print("python_ver", python_ver)
    print("phase", phase)
    _pExe, _env, env_pip, env_python = get_qtci_virtualEnv(python_ver, CI_HOST_OS, CI_HOST_ARCH, CI_TARGET_ARCH)

    if phase in ["BUILD"]:
        rmtree(_env, True)
        # Pinning the virtualenv before creating one
        python3 = "python3"
        if sys.platform == "win32":
            python3 = "python.exe"
        # Pinning the virtualenv before creating one
        run_instruction([python3, "-m", "pip", "install", "--user", "virtualenv==20.7.2"], "Failed to pin virtualenv")
        # installing to user base might not be in PATH by default.
        env_path = os.path.join(site.USER_BASE, "bin")
        v_env = os.path.join(env_path, "virtualenv")
        if sys.platform == "win32":
            env_path = os.path.join(site.USER_BASE, "Scripts")
            v_env = os.path.join(env_path, "virtualenv.exe")
        try:
            run_instruction([v_env, "--version"], "Using default virtualenv")
        except Exception as e:
            v_env = "virtualenv"
        run_instruction([v_env, "-p", _pExe,  _env], "Failed to create virtualenv")
        # When the 'python_ver' variable is empty, we are using Python 2
        try:
            print("Upgrade pip") 
            run_instruction([env_python, "--m", "pip", "install", "--upgrade", "pip"])
        except Exception as e:
            print("Failed to upgrade pip")
            pass

        run_instruction([env_pip, "install", "-r", "requirements.txt"], "Failed to install dependencies")
        if sys.platform == "win32":
            run_instruction([env_pip, "install", "numpy==1.19.3"], "Failed to install numpy 1.19.3")
        elif os.environ.get("HOST_OSVERSION_COIN") == "macos_10_13" and python_ver == "3":
            run_instruction([env_pip, "install", "numpy==1.19.4"], "Failed to install numpy")
        else:
            run_instruction([env_pip, "install", "numpy<2"], "Failed to install numpy")

    cmd = [env_python, "-u", "setup.py"]
    if phase in ["BUILD"]:
        cmd += ["build", "--standalone", "--skip-packaging"]
    elif phase in ["WHEEL"] or CI_RELEASE_CONF:
        cmd += ["bdist_wheel", "--reuse-build", "--standalone", "--skip-cmake", "--skip-make-install", "--only-package"]

    cmd += ["--build-tests",
            "--parallel=4",
            "--verbose-build"]

    if CI_TARGET_ARCH == "X86_64-ARM64":
        cmd += ["--macos-arch='x86_64;arm64'"]
        if CI_HOST_ARCH != "arm64":
            cmd += ["--macos-deployment-target=10.14"]

    if python_ver == "3":
        cmd += ["--limited-api=yes"]
    else:
        cmd += ["--skip-docs"]  # Sphinx is broken in Python 2
    if is_snapshot_build():
        cmd += ["--snapshot-build"]

    qmake_path = get_ci_qmake_path(CI_ENV_INSTALL_DIR, CI_HOST_OS)
    cmd.append(qmake_path)

    # Due to certain older CMake versions generating very long paths
    # (at least with CMake 3.6.2) when using the export() function,
    # pass the shorter paths option on Windows so we don't hit
    # the path character length limit (260).
    if CI_HOST_OS == "Windows":
        cmd += ["--shorter-paths"]

    cmd += ["--package-timestamp=" + CI_INTEGRATION_ID]

    env = os.environ
    run_instruction(cmd, "Failed to run setup.py for build", initial_env=env)

def run_build_instructions(phase):
    call_setup("3", phase)


if __name__ == "__main__":

    # Remove some environment variables that impact cmake
    arch = '32' if CI_TARGET_ARCH and CI_TARGET_ARCH == 'X86' else '64'
    # With 5.15.9 we are missing correct libclang so we need to install it for mac
    # to create universal binaries.
    if CI_HOST_OS == "MacOS" and CI_TARGET_ARCH == "X86_64-ARM64":
        provisioning()
    expand_clang_variables(arch)
    for env_var in ['CC', 'CXX']:
        if os.environ.get(env_var):
            del os.environ[env_var]

    if CI_TEST_PHASE in ["ALL", "BUILD"]:
        run_build_instructions("BUILD")

    if CI_TEST_PHASE in ["ALL", "WHEEL"]:
        run_build_instructions("WHEEL")
