# -*- coding: utf-8 -*-
"""
Determine if you have an Internet Connection.

Configuration parameters:
    cache_timeout: refresh interval for this module (default 10)
    format: display format for this module (default '{icon}')
    icon_off: show when connection is offline (default '■')
    icon_on: show when connection is online (default '●')
    timeout: time to wait for a response, in seconds (default 2)
    url: specify URL to connect when checking for a connection
        (default 'http://www.google.com')

Format placeholders:
    {icon} connection status

Color options:
    color_off: Connection offline, defaults to color_bad
    color_on: Connection online, defaults to color_good

@author obb


SAMPLE OUTPUT
{'color': '#00FF00', 'full_text': u'\u25cf'}

off
{'color': '#FF0000', 'full_text': u'\u25a0'}
"""

try:
    # python3
    from urllib.request import urlopen
except:
    # python2
    from urllib2 import urlopen


class Py3status:
    """
    """
    # available configuration parameters
    cache_timeout = 10
    format = '{icon}'
    icon_off = u'■'
    icon_on = u'●'
    timeout = 2
    url = 'http://www.google.com'

    class Meta:
        deprecated = {
            'rename': [
                {
                    'param': 'format_online',
                    'new': 'icon_on',
                    'msg': 'obsolete parameter use `icon_on`',
                },
                {
                    'param': 'format_offline',
                    'new': 'icon_off',
                    'msg': 'obsolete parameter use `icon_off`',
                },
            ],
        }

    def post_config_hook(self):
        self.color_on = self.py3.COLOR_ON or self.py3.COLOR_GOOD
        self.color_off = self.py3.COLOR_OFF or self.py3.COLOR_BAD

    def _connection_present(self):
        if '://' in self.url:
            try:
                urlopen(self.url, timeout=self.timeout)
            except:
                return False
            else:
                return True
        else:
            try:
                self.py3.command_output(['ping', '-c', '1', self.url])
            except:
                return False
            else:
                return True

    def online_status(self):
        if self._connection_present():
            icon = self.icon_on
            color = self.color_on
        else:
            color = self.color_off
            icon = self.icon_off

        return {
            'cached_until': self.py3.time_in(self.cache_timeout),
            'full_text': self.py3.safe_format(self.format, {'icon': icon}),
            'color': color
        }


if __name__ == "__main__":
    """
    Run module in test mode.
    """
    from py3status.module_test import module_test
    module_test(Py3status)
