"""
| Database (Merz) of protein backbone-backbone interactions.
| Geometries from Kenneth Merz Group, Univ. of Florida.
| Reference interaction energies from Sherrill group, Georgia Tech.
| Part of the BioFragment Database (BFDb).

- **cp**  ``'off'`` || ``'on'``

- **rlxd** ``'off'``

"""
import re
import qcdb

# <<< BBI Database Module >>>
dbse = 'BBI'

# <<< Database Members >>>
HRXN = ['004GLU-063LEU-2', '005ASP-008SER-2', '005LEU-008VAL-2', '007GLN-010ASN-1', '007VAL-041VAL-1',
        '008ILE-012LEU-1', '008TRP-251HIE-2', '011ILE-014PHE-1', '012LEU-085ASP-1', '014LYS-018GLU-1',
        '014VAL-017GLU-1', '015GLN-018LEU-1', '015LEU-026LEU-1', '017ASN-021LYS-2', '017LEU-025ILE-1',
        '019LEU-022MET-1', '023LYS-146SER-2', '024PHE-028ALA-1', '024THR-041GLU-2', '025GLY-029SER-2',
        '025ILE-070TRP-1', '027VAL-031LEU-1', '027VAL-068GLU-2', '029GLU-032VAL-1', '030TRP-178GLY-2',
        '033ASN-036TRP-2', '033VAL-115VAL-1', '035TRP-056ILE-1', '037PRO-041ALA-1', '038ALA-041GLY-1',
        '038GLU-047GLU-2', '039SER-046MET-1', '040THR-048VAL-2', '040THR-049GLN-2', '041GLY-045LEU-2',
        '041LEU-045LEU-1', '042LEU-047ILE-2', '044SER-047GLU-1', '044TRP-054LEU-1', '048GLU-052LYS-1',
        '051ALA-054VAL-1', '051ALA-055ASN-1', '052CYS-056ALA-1', '054ARG-062GLN-1', '055ASN-058GLU-2',
        '060LEU-064TYR-1', '060TYR-064LEU-2', '061VAL-064TYR-2', '062LEU-066LYS-2', '064GLN-067ARG-1',
        '064TYR-067GLU-1', '066PHE-072THR-1', '068ASP-072GLN-1', '071GLU-075ASP-1', '072ASN-075ARG-2',
        '072THR-075PRO-1', '081ASN-084GLY-1', '081LEU-084LYS-1', '082LEU-106LEU-2', '084LEU-088MET-2',
        '087ALA-171PRO-2', '087LEU-090TYR-1', '088PHE-091ALA-1', '089MET-093GLY-1', '092SER-096ARG-2',
        '095GLN-183ILE-1', '095LYS-107ILE-2', '097GLU-100THR-2', '102GLN-106ILE-1', '104VAL-108ILE-1',
        '108LYS-112TYR-1', '108TYR-129TRP-1', '112SER-115ALA-2', '112TRP-115ARG-1', '113TRP-124HIE-2',
        '115GLN-118ARG-2', '119MET-122VAL-1', '121LYS-125ALA-1', '121VAL-155THR-2', '126VAL-129ALA-1',
        '133LYS-137ASN-1', '134GLU-138ARG-1', '135ARG-152ALA-2', '137SER-144LYS-1', '140SER-144THR-2',
        '142ALA-146PHE-1', '143VAL-147GLU-1', '146PHE-150LEU-1', '150LYS-158LEU-2', '157LYS-160VAL-1',
        '162ALA-176GLY-1', '167GLY-232GLY-1', '168GLY-172LYS-2', '171ALA-175GLU-1', '172GLY-175TRP-1',
        '197TYR-201LEU-2', '199SER-202TYR-1', '205THR-208GLU-1', '205THR-209LEU-2', '228ARG-232GLU-1']
HRXN_SHB = [
        '004GLU-063LEU-2', '005LEU-008VAL-2', '007VAL-041VAL-1', '008ILE-012LEU-1', '008TRP-251HIE-2',
        '012LEU-085ASP-1', '014LYS-018GLU-1', '014VAL-017GLU-1', '015LEU-026LEU-1', '017ASN-021LYS-2',
        '017LEU-025ILE-1', '019LEU-022MET-1', '023LYS-146SER-2', '024PHE-028ALA-1', '024THR-041GLU-2',
        '025GLY-029SER-2', '025ILE-070TRP-1', '027VAL-031LEU-1', '027VAL-068GLU-2', '030TRP-178GLY-2',
        '033ASN-036TRP-2', '033VAL-115VAL-1', '035TRP-056ILE-1', '037PRO-041ALA-1', '038GLU-047GLU-2',
        '039SER-046MET-1', '040THR-048VAL-2', '040THR-049GLN-2', '041GLY-045LEU-2', '041LEU-045LEU-1',
        '042LEU-047ILE-2', '044TRP-054LEU-1', '048GLU-052LYS-1', '051ALA-054VAL-1', '051ALA-055ASN-1',
        '052CYS-056ALA-1', '054ARG-062GLN-1', '055ASN-058GLU-2', '060LEU-064TYR-1', '060TYR-064LEU-2',
        '061VAL-064TYR-2', '062LEU-066LYS-2', '064TYR-067GLU-1', '068ASP-072GLN-1', '071GLU-075ASP-1',
        '081ASN-084GLY-1', '082LEU-106LEU-2', '084LEU-088MET-2', '087ALA-171PRO-2', '087LEU-090TYR-1',
        '089MET-093GLY-1', '092SER-096ARG-2', '095GLN-183ILE-1', '095LYS-107ILE-2', '097GLU-100THR-2',
        '102GLN-106ILE-1', '104VAL-108ILE-1', '108LYS-112TYR-1', '108TYR-129TRP-1', '113TRP-124HIE-2',
        '115GLN-118ARG-2', '121LYS-125ALA-1', '121VAL-155THR-2', '133LYS-137ASN-1', '134GLU-138ARG-1',
        '135ARG-152ALA-2', '137SER-144LYS-1', '140SER-144THR-2', '142ALA-146PHE-1', '143VAL-147GLU-1',
        '146PHE-150LEU-1', '150LYS-158LEU-2', '157LYS-160VAL-1', '162ALA-176GLY-1', '167GLY-232GLY-1',
        '168GLY-172LYS-2', '172GLY-175TRP-1', '197TYR-201LEU-2', '205THR-209LEU-2', '228ARG-232GLU-1']
HRXN_UA = [
        '005ASP-008SER-2', '007GLN-010ASN-1', '011ILE-014PHE-1', '015GLN-018LEU-1', '029GLU-032VAL-1',
        '038ALA-041GLY-1', '044SER-047GLU-1', '064GLN-067ARG-1', '066PHE-072THR-1', '072ASN-075ARG-2',
        '072THR-075PRO-1', '081LEU-084LYS-1', '088PHE-091ALA-1', '112SER-115ALA-2', '112TRP-115ARG-1',
        '119MET-122VAL-1', '126VAL-129ALA-1', '171ALA-175GLU-1', '199SER-202TYR-1', '205THR-208GLU-1']
HRXN_BBI25 = [
        '007VAL-041VAL-1', '008TRP-251HIE-2', '012LEU-085ASP-1', '015LEU-026LEU-1', '017ASN-021LYS-2',
        '030TRP-178GLY-2', '033ASN-036TRP-2', '033VAL-115VAL-1', '038ALA-041GLY-1', '040THR-049GLN-2',
        '041GLY-045LEU-2', '044SER-047GLU-1', '044TRP-054LEU-1', '061VAL-064TYR-2', '072THR-075PRO-1',
        '081LEU-084LYS-1', '084LEU-088MET-2', '087LEU-090TYR-1', '108LYS-112TYR-1', '108TYR-129TRP-1',
        '112SER-115ALA-2', '121VAL-155THR-2', '171ALA-175GLU-1', '197TYR-201LEU-2', '228ARG-232GLU-1']

# <<< Chemical Systems Involved >>>
RXNM = {}     # reaction matrix of reagent contributions per reaction
ACTV = {}     # order of active reagents per reaction
ACTV_CP = {}  # order of active reagents per counterpoise-corrected reaction
ACTV_SA = {}  # order of active reagents for non-supermolecular calculations
for rxn in HRXN:

    RXNM[   '%s-%s' % (dbse, rxn)] = {'%s-%s-dimer'      % (dbse, rxn) : +1,
                                      '%s-%s-monoA-CP'   % (dbse, rxn) : -1,
                                      '%s-%s-monoB-CP'   % (dbse, rxn) : -1,
                                      '%s-%s-monoA-unCP' % (dbse, rxn) : -1,
                                      '%s-%s-monoB-unCP' % (dbse, rxn) : -1 }

    ACTV_SA['%s-%s' % (dbse, rxn)] = ['%s-%s-dimer'      % (dbse, rxn) ]

    ACTV_CP['%s-%s' % (dbse, rxn)] = ['%s-%s-dimer'      % (dbse, rxn),
                                      '%s-%s-monoA-CP'   % (dbse, rxn),
                                      '%s-%s-monoB-CP'   % (dbse, rxn) ]

    ACTV[   '%s-%s' % (dbse, rxn)] = ['%s-%s-dimer'      % (dbse, rxn),
                                      '%s-%s-monoA-unCP' % (dbse, rxn),
                                      '%s-%s-monoB-unCP' % (dbse, rxn) ]

# <<< Reference Values [kcal/mol] >>>
BIND = {}
# Bronze
BIND_BRONZE = {}
BIND_BRONZE['%s-%s' % (dbse, '004GLU-063LEU-2')] =   -8.393
BIND_BRONZE['%s-%s' % (dbse, '005ASP-008SER-2')] =   -1.518
BIND_BRONZE['%s-%s' % (dbse, '005LEU-008VAL-2')] =   -6.233
BIND_BRONZE['%s-%s' % (dbse, '007GLN-010ASN-1')] =   -1.442
BIND_BRONZE['%s-%s' % (dbse, '007VAL-041VAL-1')] =   -6.114
BIND_BRONZE['%s-%s' % (dbse, '008ILE-012LEU-1')] =   -7.172
BIND_BRONZE['%s-%s' % (dbse, '008TRP-251HIE-2')] =   -7.615
BIND_BRONZE['%s-%s' % (dbse, '011ILE-014PHE-1')] =   -1.098
BIND_BRONZE['%s-%s' % (dbse, '012LEU-085ASP-1')] =   -8.072
BIND_BRONZE['%s-%s' % (dbse, '014LYS-018GLU-1')] =   -7.073
BIND_BRONZE['%s-%s' % (dbse, '014VAL-017GLU-1')] =   -7.550
BIND_BRONZE['%s-%s' % (dbse, '015GLN-018LEU-1')] =   -1.272
BIND_BRONZE['%s-%s' % (dbse, '015LEU-026LEU-1')] =   -7.406
BIND_BRONZE['%s-%s' % (dbse, '017ASN-021LYS-2')] =   -6.291
BIND_BRONZE['%s-%s' % (dbse, '017LEU-025ILE-1')] =   -6.664
BIND_BRONZE['%s-%s' % (dbse, '019LEU-022MET-1')] =   -5.842
BIND_BRONZE['%s-%s' % (dbse, '023LYS-146SER-2')] =   -7.712
BIND_BRONZE['%s-%s' % (dbse, '024PHE-028ALA-1')] =   -7.167
BIND_BRONZE['%s-%s' % (dbse, '024THR-041GLU-2')] =   -7.294
BIND_BRONZE['%s-%s' % (dbse, '025GLY-029SER-2')] =   -6.834
BIND_BRONZE['%s-%s' % (dbse, '025ILE-070TRP-1')] =   -7.910
BIND_BRONZE['%s-%s' % (dbse, '027VAL-031LEU-1')] =   -7.772
BIND_BRONZE['%s-%s' % (dbse, '027VAL-068GLU-2')] =   -7.712
BIND_BRONZE['%s-%s' % (dbse, '029GLU-032VAL-1')] =   -1.399
BIND_BRONZE['%s-%s' % (dbse, '030TRP-178GLY-2')] =   -8.039
BIND_BRONZE['%s-%s' % (dbse, '033ASN-036TRP-2')] =    1.328
BIND_BRONZE['%s-%s' % (dbse, '033VAL-115VAL-1')] =   -7.858
BIND_BRONZE['%s-%s' % (dbse, '035TRP-056ILE-1')] =   -7.840
BIND_BRONZE['%s-%s' % (dbse, '037PRO-041ALA-1')] =   -6.672
BIND_BRONZE['%s-%s' % (dbse, '038ALA-041GLY-1')] =   -1.134
BIND_BRONZE['%s-%s' % (dbse, '038GLU-047GLU-2')] =   -4.753
BIND_BRONZE['%s-%s' % (dbse, '039SER-046MET-1')] =   -7.258
BIND_BRONZE['%s-%s' % (dbse, '040THR-048VAL-2')] =   -7.407
BIND_BRONZE['%s-%s' % (dbse, '040THR-049GLN-2')] =   -8.172
BIND_BRONZE['%s-%s' % (dbse, '041GLY-045LEU-2')] =   -7.460
BIND_BRONZE['%s-%s' % (dbse, '041LEU-045LEU-1')] =   -7.310
BIND_BRONZE['%s-%s' % (dbse, '042LEU-047ILE-2')] =   -7.191
BIND_BRONZE['%s-%s' % (dbse, '044SER-047GLU-1')] =   -2.585
BIND_BRONZE['%s-%s' % (dbse, '044TRP-054LEU-1')] =   -7.819
BIND_BRONZE['%s-%s' % (dbse, '048GLU-052LYS-1')] =   -6.567
BIND_BRONZE['%s-%s' % (dbse, '051ALA-054VAL-1')] =    2.681
BIND_BRONZE['%s-%s' % (dbse, '051ALA-055ASN-1')] =   -6.170
BIND_BRONZE['%s-%s' % (dbse, '052CYS-056ALA-1')] =   -6.957
BIND_BRONZE['%s-%s' % (dbse, '054ARG-062GLN-1')] =   -6.344
BIND_BRONZE['%s-%s' % (dbse, '055ASN-058GLU-2')] =   -5.542
BIND_BRONZE['%s-%s' % (dbse, '060LEU-064TYR-1')] =   -6.335
BIND_BRONZE['%s-%s' % (dbse, '060TYR-064LEU-2')] =   -7.322
BIND_BRONZE['%s-%s' % (dbse, '061VAL-064TYR-2')] =   -5.629
BIND_BRONZE['%s-%s' % (dbse, '062LEU-066LYS-2')] =   -7.169
BIND_BRONZE['%s-%s' % (dbse, '064GLN-067ARG-1')] =   -1.442
BIND_BRONZE['%s-%s' % (dbse, '064TYR-067GLU-1')] =   -6.343
BIND_BRONZE['%s-%s' % (dbse, '066PHE-072THR-1')] =   -1.684
BIND_BRONZE['%s-%s' % (dbse, '068ASP-072GLN-1')] =   -3.610
BIND_BRONZE['%s-%s' % (dbse, '071GLU-075ASP-1')] =   -7.049
BIND_BRONZE['%s-%s' % (dbse, '072ASN-075ARG-2')] =   -1.244
BIND_BRONZE['%s-%s' % (dbse, '072THR-075PRO-1')] =   -2.038
BIND_BRONZE['%s-%s' % (dbse, '081ASN-084GLY-1')] =   -5.923
BIND_BRONZE['%s-%s' % (dbse, '081LEU-084LYS-1')] =   -1.442
BIND_BRONZE['%s-%s' % (dbse, '082LEU-106LEU-2')] =   -7.618
BIND_BRONZE['%s-%s' % (dbse, '084LEU-088MET-2')] =   -7.225
BIND_BRONZE['%s-%s' % (dbse, '087ALA-171PRO-2')] =   -8.151
BIND_BRONZE['%s-%s' % (dbse, '087LEU-090TYR-1')] =   -6.068
BIND_BRONZE['%s-%s' % (dbse, '088PHE-091ALA-1')] =   -1.387
BIND_BRONZE['%s-%s' % (dbse, '089MET-093GLY-1')] =   -6.694
BIND_BRONZE['%s-%s' % (dbse, '092SER-096ARG-2')] =   -7.368
BIND_BRONZE['%s-%s' % (dbse, '095GLN-183ILE-1')] =   -8.222
BIND_BRONZE['%s-%s' % (dbse, '095LYS-107ILE-2')] =   -7.781
BIND_BRONZE['%s-%s' % (dbse, '097GLU-100THR-2')] =   -5.213
BIND_BRONZE['%s-%s' % (dbse, '102GLN-106ILE-1')] =   -6.443
BIND_BRONZE['%s-%s' % (dbse, '104VAL-108ILE-1')] =   -7.408
BIND_BRONZE['%s-%s' % (dbse, '108LYS-112TYR-1')] =   -7.210
BIND_BRONZE['%s-%s' % (dbse, '108TYR-129TRP-1')] =   -7.356
BIND_BRONZE['%s-%s' % (dbse, '112SER-115ALA-2')] =   -1.716
BIND_BRONZE['%s-%s' % (dbse, '112TRP-115ARG-1')] =   -1.971
BIND_BRONZE['%s-%s' % (dbse, '113TRP-124HIE-2')] =   -7.921
BIND_BRONZE['%s-%s' % (dbse, '115GLN-118ARG-2')] =   -6.439
BIND_BRONZE['%s-%s' % (dbse, '119MET-122VAL-1')] =   -1.319
BIND_BRONZE['%s-%s' % (dbse, '121LYS-125ALA-1')] =   -5.983
BIND_BRONZE['%s-%s' % (dbse, '121VAL-155THR-2')] =   -7.257
BIND_BRONZE['%s-%s' % (dbse, '126VAL-129ALA-1')] =   -1.269
BIND_BRONZE['%s-%s' % (dbse, '133LYS-137ASN-1')] =   -7.240
BIND_BRONZE['%s-%s' % (dbse, '134GLU-138ARG-1')] =   -7.140
BIND_BRONZE['%s-%s' % (dbse, '135ARG-152ALA-2')] =   -7.124
BIND_BRONZE['%s-%s' % (dbse, '137SER-144LYS-1')] =   -7.506
BIND_BRONZE['%s-%s' % (dbse, '140SER-144THR-2')] =   -7.447
BIND_BRONZE['%s-%s' % (dbse, '142ALA-146PHE-1')] =   -7.431
BIND_BRONZE['%s-%s' % (dbse, '143VAL-147GLU-1')] =   -7.213
BIND_BRONZE['%s-%s' % (dbse, '146PHE-150LEU-1')] =   -7.044
BIND_BRONZE['%s-%s' % (dbse, '150LYS-158LEU-2')] =   -8.308
BIND_BRONZE['%s-%s' % (dbse, '157LYS-160VAL-1')] =    2.056
BIND_BRONZE['%s-%s' % (dbse, '162ALA-176GLY-1')] =   -8.225
BIND_BRONZE['%s-%s' % (dbse, '167GLY-232GLY-1')] =   -6.892
BIND_BRONZE['%s-%s' % (dbse, '168GLY-172LYS-2')] =   -7.259
BIND_BRONZE['%s-%s' % (dbse, '171ALA-175GLU-1')] =   -1.671
BIND_BRONZE['%s-%s' % (dbse, '172GLY-175TRP-1')] =   -5.146
BIND_BRONZE['%s-%s' % (dbse, '197TYR-201LEU-2')] =   -6.637
BIND_BRONZE['%s-%s' % (dbse, '199SER-202TYR-1')] =   -1.087
BIND_BRONZE['%s-%s' % (dbse, '205THR-208GLU-1')] =   -1.028
BIND_BRONZE['%s-%s' % (dbse, '205THR-209LEU-2')] =   -5.627
BIND_BRONZE['%s-%s' % (dbse, '228ARG-232GLU-1')] =   -7.350
# Silver
BIND_SILVER = {}
BIND_SILVER['%s-%s' % (dbse, '004GLU-063LEU-2')] =   -8.623
BIND_SILVER['%s-%s' % (dbse, '005ASP-008SER-2')] =   -1.589
BIND_SILVER['%s-%s' % (dbse, '005LEU-008VAL-2')] =   -6.388
BIND_SILVER['%s-%s' % (dbse, '007GLN-010ASN-1')] =   -1.510
BIND_SILVER['%s-%s' % (dbse, '007VAL-041VAL-1')] =   -6.366
BIND_SILVER['%s-%s' % (dbse, '008ILE-012LEU-1')] =   -7.410
BIND_SILVER['%s-%s' % (dbse, '008TRP-251HIE-2')] =   -7.771
BIND_SILVER['%s-%s' % (dbse, '011ILE-014PHE-1')] =   -1.180
BIND_SILVER['%s-%s' % (dbse, '012LEU-085ASP-1')] =   -8.281
BIND_SILVER['%s-%s' % (dbse, '014LYS-018GLU-1')] =   -7.312
BIND_SILVER['%s-%s' % (dbse, '014VAL-017GLU-1')] =   -7.767
BIND_SILVER['%s-%s' % (dbse, '015GLN-018LEU-1')] =   -1.348
BIND_SILVER['%s-%s' % (dbse, '015LEU-026LEU-1')] =   -7.651
BIND_SILVER['%s-%s' % (dbse, '017ASN-021LYS-2')] =   -6.497
BIND_SILVER['%s-%s' % (dbse, '017LEU-025ILE-1')] =   -6.884
BIND_SILVER['%s-%s' % (dbse, '019LEU-022MET-1')] =   -5.970
BIND_SILVER['%s-%s' % (dbse, '023LYS-146SER-2')] =   -7.922
BIND_SILVER['%s-%s' % (dbse, '024PHE-028ALA-1')] =   -7.448
BIND_SILVER['%s-%s' % (dbse, '024THR-041GLU-2')] =   -7.496
BIND_SILVER['%s-%s' % (dbse, '025GLY-029SER-2')] =   -7.075
BIND_SILVER['%s-%s' % (dbse, '025ILE-070TRP-1')] =   -8.120
BIND_SILVER['%s-%s' % (dbse, '027VAL-031LEU-1')] =   -8.041
BIND_SILVER['%s-%s' % (dbse, '027VAL-068GLU-2')] =   -7.943
BIND_SILVER['%s-%s' % (dbse, '029GLU-032VAL-1')] =   -1.472
BIND_SILVER['%s-%s' % (dbse, '030TRP-178GLY-2')] =   -8.247
BIND_SILVER['%s-%s' % (dbse, '033ASN-036TRP-2')] =    1.229
BIND_SILVER['%s-%s' % (dbse, '033VAL-115VAL-1')] =   -8.073
BIND_SILVER['%s-%s' % (dbse, '035TRP-056ILE-1')] =   -8.082
BIND_SILVER['%s-%s' % (dbse, '037PRO-041ALA-1')] =   -6.917
BIND_SILVER['%s-%s' % (dbse, '038ALA-041GLY-1')] =   -1.191
BIND_SILVER['%s-%s' % (dbse, '038GLU-047GLU-2')] =   -4.865
BIND_SILVER['%s-%s' % (dbse, '039SER-046MET-1')] =   -7.468
BIND_SILVER['%s-%s' % (dbse, '040THR-048VAL-2')] =   -7.639
BIND_SILVER['%s-%s' % (dbse, '040THR-049GLN-2')] =   -8.407
BIND_SILVER['%s-%s' % (dbse, '041GLY-045LEU-2')] =   -7.697
BIND_SILVER['%s-%s' % (dbse, '041LEU-045LEU-1')] =   -7.561
BIND_SILVER['%s-%s' % (dbse, '042LEU-047ILE-2')] =   -7.426
BIND_SILVER['%s-%s' % (dbse, '044SER-047GLU-1')] =   -2.638
BIND_SILVER['%s-%s' % (dbse, '044TRP-054LEU-1')] =   -8.043
BIND_SILVER['%s-%s' % (dbse, '048GLU-052LYS-1')] =   -6.828
BIND_SILVER['%s-%s' % (dbse, '051ALA-054VAL-1')] =    2.680
BIND_SILVER['%s-%s' % (dbse, '051ALA-055ASN-1')] =   -6.380
BIND_SILVER['%s-%s' % (dbse, '052CYS-056ALA-1')] =   -7.166
BIND_SILVER['%s-%s' % (dbse, '054ARG-062GLN-1')] =   -6.516
BIND_SILVER['%s-%s' % (dbse, '055ASN-058GLU-2')] =   -5.672
BIND_SILVER['%s-%s' % (dbse, '060LEU-064TYR-1')] =   -6.531
BIND_SILVER['%s-%s' % (dbse, '060TYR-064LEU-2')] =   -7.576
BIND_SILVER['%s-%s' % (dbse, '061VAL-064TYR-2')] =   -5.751
BIND_SILVER['%s-%s' % (dbse, '062LEU-066LYS-2')] =   -7.396
BIND_SILVER['%s-%s' % (dbse, '064GLN-067ARG-1')] =   -1.502
BIND_SILVER['%s-%s' % (dbse, '064TYR-067GLU-1')] =   -6.519
BIND_SILVER['%s-%s' % (dbse, '066PHE-072THR-1')] =   -1.963
BIND_SILVER['%s-%s' % (dbse, '068ASP-072GLN-1')] =   -3.899
BIND_SILVER['%s-%s' % (dbse, '071GLU-075ASP-1')] =   -7.283
BIND_SILVER['%s-%s' % (dbse, '072ASN-075ARG-2')] =   -1.300
BIND_SILVER['%s-%s' % (dbse, '072THR-075PRO-1')] =   -2.100
BIND_SILVER['%s-%s' % (dbse, '081ASN-084GLY-1')] =   -5.985
BIND_SILVER['%s-%s' % (dbse, '081LEU-084LYS-1')] =   -1.507
BIND_SILVER['%s-%s' % (dbse, '082LEU-106LEU-2')] =   -7.836
BIND_SILVER['%s-%s' % (dbse, '084LEU-088MET-2')] =   -7.448
BIND_SILVER['%s-%s' % (dbse, '087ALA-171PRO-2')] =   -8.365
BIND_SILVER['%s-%s' % (dbse, '087LEU-090TYR-1')] =   -6.187
BIND_SILVER['%s-%s' % (dbse, '088PHE-091ALA-1')] =   -1.445
BIND_SILVER['%s-%s' % (dbse, '089MET-093GLY-1')] =   -6.946
BIND_SILVER['%s-%s' % (dbse, '092SER-096ARG-2')] =   -7.589
BIND_SILVER['%s-%s' % (dbse, '095GLN-183ILE-1')] =   -8.448
BIND_SILVER['%s-%s' % (dbse, '095LYS-107ILE-2')] =   -8.007
BIND_SILVER['%s-%s' % (dbse, '097GLU-100THR-2')] =   -5.303
BIND_SILVER['%s-%s' % (dbse, '102GLN-106ILE-1')] =   -6.676
BIND_SILVER['%s-%s' % (dbse, '104VAL-108ILE-1')] =   -7.628
BIND_SILVER['%s-%s' % (dbse, '108LYS-112TYR-1')] =   -7.397
BIND_SILVER['%s-%s' % (dbse, '108TYR-129TRP-1')] =   -7.591
BIND_SILVER['%s-%s' % (dbse, '112SER-115ALA-2')] =   -1.785
BIND_SILVER['%s-%s' % (dbse, '112TRP-115ARG-1')] =   -2.022
BIND_SILVER['%s-%s' % (dbse, '113TRP-124HIE-2')] =   -8.153
BIND_SILVER['%s-%s' % (dbse, '115GLN-118ARG-2')] =   -6.585
BIND_SILVER['%s-%s' % (dbse, '119MET-122VAL-1')] =   -1.388
BIND_SILVER['%s-%s' % (dbse, '121LYS-125ALA-1')] =   -6.188
BIND_SILVER['%s-%s' % (dbse, '121VAL-155THR-2')] =   -7.480
BIND_SILVER['%s-%s' % (dbse, '126VAL-129ALA-1')] =   -1.340
BIND_SILVER['%s-%s' % (dbse, '133LYS-137ASN-1')] =   -7.522
BIND_SILVER['%s-%s' % (dbse, '134GLU-138ARG-1')] =   -7.375
BIND_SILVER['%s-%s' % (dbse, '135ARG-152ALA-2')] =   -7.377
BIND_SILVER['%s-%s' % (dbse, '137SER-144LYS-1')] =   -7.731
BIND_SILVER['%s-%s' % (dbse, '140SER-144THR-2')] =   -7.668
BIND_SILVER['%s-%s' % (dbse, '142ALA-146PHE-1')] =   -7.653
BIND_SILVER['%s-%s' % (dbse, '143VAL-147GLU-1')] =   -7.454
BIND_SILVER['%s-%s' % (dbse, '146PHE-150LEU-1')] =   -7.290
BIND_SILVER['%s-%s' % (dbse, '150LYS-158LEU-2')] =   -8.530
BIND_SILVER['%s-%s' % (dbse, '157LYS-160VAL-1')] =    1.996
BIND_SILVER['%s-%s' % (dbse, '162ALA-176GLY-1')] =   -8.443
BIND_SILVER['%s-%s' % (dbse, '167GLY-232GLY-1')] =   -7.120
BIND_SILVER['%s-%s' % (dbse, '168GLY-172LYS-2')] =   -7.440
BIND_SILVER['%s-%s' % (dbse, '171ALA-175GLU-1')] =   -1.912
BIND_SILVER['%s-%s' % (dbse, '172GLY-175TRP-1')] =   -5.297
BIND_SILVER['%s-%s' % (dbse, '197TYR-201LEU-2')] =   -6.934
BIND_SILVER['%s-%s' % (dbse, '199SER-202TYR-1')] =   -1.140
BIND_SILVER['%s-%s' % (dbse, '205THR-208GLU-1')] =   -1.083
BIND_SILVER['%s-%s' % (dbse, '205THR-209LEU-2')] =   -5.867
BIND_SILVER['%s-%s' % (dbse, '228ARG-232GLU-1')] =   -7.561
# Set default
BIND = BIND_SILVER

# <<< Comment Lines >>>
TAGL = {}
rxnpattern = re.compile(r'^(.+)-(.+)-(.+)$')
for rxn in HRXN:
    molname = rxnpattern.match(rxn)
    TAGL['%s-%s'            % (dbse, rxn)] = """Residue %s and %s interaction No. %s""" % (molname.group(1), molname.group(2), molname.group(3))
    TAGL['%s-%s-dimer'      % (dbse, rxn)] = """Dimer from %s""" % (rxn)
    TAGL['%s-%s-monoA-CP'   % (dbse, rxn)] = """Monomer A from %s""" % (rxn)
    TAGL['%s-%s-monoB-CP'   % (dbse, rxn)] = """Monomer B from %s""" % (rxn)
    TAGL['%s-%s-monoA-unCP' % (dbse, rxn)] = """Monomer A from %s""" % (rxn)
    TAGL['%s-%s-monoB-unCP' % (dbse, rxn)] = """Monomer B from %s""" % (rxn)

TAGL['dbse'] = 'interaction energies for peptide backbone-backbone complexes'
TAGL['shb'] = 'singly hydrogen-bonded complexes'
TAGL['ua'] = 'unaligned backbone structures'
TAGL['bbi25'] = 'representative sample of 25'
TAGL['default'] = 'entire database'

# <<< Geometry Specification Strings >>>
GEOS = {}

GEOS['%s-%s-dimer' % (dbse, '004GLU-063LEU-2')] = qcdb.Molecule("""
0 1
C       31.01400000    -3.84200000     7.91700000
C       32.77900000    -5.82000000    10.65500000
C       32.37100000    -4.63300000     9.78100000
O       32.83000000    -3.47700000     9.99000000
N       31.53500000    -4.90600000     8.77100000
H       31.20900000    -5.85900000     8.62600000
H       31.29400000    -2.86400000     8.30900000
H       32.22100000    -6.71200000    10.37200000
H       31.43191000    -3.93711000     6.90393200
H       29.91523000    -3.88500500     7.88761400
H       33.86081000    -5.95898200    10.51226000
H       32.57115000    -5.57809500    11.70775000
--
0 1
C       29.64500000    -8.60100000     6.64000000
C       31.49500000   -10.17100000     9.56100000
C       30.48100000    -8.70100000     7.90600000
O       30.84300000    -7.68500000     8.51100000
N       30.69600000    -9.92800000     8.36400000
H       29.35100000    -9.59600000     6.30500000
H       30.32100000   -10.72500000     7.85800000
H       32.01200000    -9.26900000     9.87800000
H       28.75868000    -8.00110100     6.89409300
H       30.21876000    -8.10158000     5.84540300
H       30.83944000   -10.49806000    10.38153000
H       32.23175000   -10.94459000     9.29880200
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '005ASP-008SER-2')] = qcdb.Molecule("""
0 1
C       26.24300000    -6.23800000    -1.26400000
C       23.28800000    -8.18300000     0.13400000
C       25.18600000    -6.71300000    -0.27500000
O       25.16800000    -6.30000000     0.88000000
N       24.31100000    -7.61000000    -0.72800000
H       27.04400000    -5.75900000    -0.69800000
H       24.33700000    -7.87600000    -1.70700000
H       22.67200000    -7.36900000     0.51900000
H       26.66298000    -7.09677700    -1.80816800
H       25.82028000    -5.49661700    -1.95801400
H       22.64828000    -8.87143800    -0.43766880
H       23.74797000    -8.69263800     0.99347450
--
0 1
C       27.48200000    -7.33500000     3.49700000
C       25.64400000   -10.39800000     2.15900000
C       26.29300000    -9.42100000     3.13600000
O       26.27700000    -9.65100000     4.34600000
N       26.85700000    -8.33100000     2.62700000
H       26.86500000    -8.18600000     1.62300000
H       28.19300000    -7.85100000     4.14400000
H       24.90900000   -10.95900000     2.73100000
H       28.05180000    -6.59511600     2.91571300
H       26.74688000    -6.83531700     4.14500400
H       25.11716000    -9.88318200     1.34205500
H       26.38917000   -11.10225000     1.76057000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '005LEU-008VAL-2')] = qcdb.Molecule("""
0 1
C       14.16900000    62.02000000     7.59200000
C       12.42900000    58.71800000     8.29000000
C       13.07200000    61.01200000     7.90200000
O       11.95100000    61.39900000     8.24000000
N       13.36700000    59.71700000     7.80200000
H       14.58400000    62.25800000     8.57100000
H       14.29300000    59.43200000     7.51100000
H       11.51200000    58.78500000     7.70400000
H       14.95972000    61.61525000     6.94320200
H       13.74934000    62.95496000     7.19232400
H       12.84693000    57.70447000     8.20008400
H       12.16339000    58.94186000     9.33371500
--
0 1
C       10.16200000    62.96600000    11.02600000
C       10.21400000    59.16000000    11.38700000
C       10.05400000    60.64000000    11.75100000
O        9.52600000    60.96500000    12.81800000
N       10.48900000    61.54600000    10.88600000
H       10.93600000    61.23700000    10.03200000
H       10.21200000    63.26400000    12.07400000
H        9.18900000    58.78200000    11.38300000
H       10.88144000    63.57503000    10.45900000
H        9.13582400    63.11971000    10.66085000
H       10.63054000    58.95248000    10.39029000
H       10.74514000    58.61961000    12.18442000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '007GLN-010ASN-1')] = qcdb.Molecule("""
0 1
C        2.71600000     0.82000000    22.90400000
C        6.42600000     1.29500000    22.11000000
C        4.16200000     0.46200000    22.57800000
O        4.51600000    -0.71700000    22.51100000
N        4.99800000     1.48000000    22.38400000
H        2.22900000    -0.07200000    23.30100000
H        4.62300000     2.42400000    22.39500000
H        6.53500000     0.63900000    21.24300000
H        2.66950800     1.62761900    23.64938000
H        2.21975900     1.11228600    21.96682000
H        6.89564800     2.25983500    21.86808000
H        6.93538300     0.79349740    22.94608000
--
0 1
C        5.29400000    -2.80800000    25.39800000
C        7.40700000     0.34000000    25.70300000
C        7.05100000    -1.14600000    25.69200000
O        7.88300000    -2.00700000    25.99000000
N        5.79600000    -1.44000000    25.36200000
H        5.14500000    -0.68000000    25.19200000
H        5.59200000    -3.26200000    26.34600000
H        8.49600000     0.41700000    25.70700000
H        4.19525100    -2.81731800    25.34640000
H        5.73195000    -3.44291900    24.61373000
H        6.99946900     0.84474080    24.81466000
H        7.00958500     0.77023540    26.63411000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '007VAL-041VAL-1')] = qcdb.Molecule("""
0 1
C       22.55700000     0.21000000     5.45400000
C       19.77500000     1.94900000     3.54200000
C       21.87500000     0.97900000     4.30300000
O       22.41500000     1.19200000     3.20900000
N       20.64000000     1.38900000     4.57200000
H       21.83700000    -0.53400000     5.80400000
H       20.24700000     1.19900000     5.49200000
H       19.82400000     1.35600000     2.62700000
H       23.47928000    -0.28719010     5.11903500
H       22.73621000     0.90317170     6.28910400
H       18.76701000     1.96809900     3.98198100
H       20.08484000     2.97985500     3.31541500
--
0 1
C       23.51900000     1.36800000    -0.32800000
C       25.20800000     3.08800000     2.65500000
C       25.00500000     2.50600000     1.27600000
O       25.94600000     2.48800000     0.50700000
N       23.81700000     1.99500000     0.97100000
H       23.10500000     1.94600000     1.69300000
H       24.38600000     1.40300000    -0.99000000
H       24.55300000     2.55600000     3.34200000
H       23.23073000     0.31617780    -0.18456970
H       22.69658000     1.94062500    -0.78157200
H       26.25647000     2.90498000     2.93286400
H       24.94375000     4.15420700     2.71309200
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '008ILE-012LEU-1')] = qcdb.Molecule("""
0 1
C       32.59400000    -4.62600000    57.97600000
C       33.24900000    -3.95000000    54.28700000
C       32.35300000    -4.27000000    56.51000000
O       31.20800000    -4.13900000    56.08400000
N       33.42400000    -4.16200000    55.71600000
H       31.64400000    -4.97300000    58.38600000
H       34.36000000    -4.23300000    56.10400000
H       32.67300000    -3.03700000    54.12300000
H       33.33891000    -5.43143000    58.05595000
H       32.92266000    -3.75266200    58.55846000
H       34.21676000    -3.84151300    53.77546000
H       32.68009000    -4.76497700    53.81566000
--
0 1
C       28.25400000    -3.45600000    53.71400000
C       28.74800000    -6.84300000    55.36600000
C       28.07800000    -5.67300000    54.64800000
O       26.91000000    -5.77200000    54.26300000
N       28.79000000    -4.56300000    54.47700000
H       29.72100000    -4.49600000    54.87400000
H       27.34700000    -3.13500000    54.22400000
H       28.21100000    -7.74000000    55.06900000
H       28.99593000    -2.64388400    53.71543000
H       27.96856000    -3.80399000    52.71029000
H       29.80490000    -6.93581600    55.07558000
H       28.63932000    -6.73144500    56.45492000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '008TRP-251HIE-2')] = qcdb.Molecule("""
0 1
C       27.44300000    25.35600000    25.79500000
C       27.24800000    25.85300000    29.56200000
C       27.86800000    25.37900000    27.24700000
O       29.06200000    25.29900000    27.53400000
N       26.91100000    25.57500000    28.15500000
H       26.36200000    25.50200000    25.73300000
H       25.94500000    25.63300000    27.85800000
H       28.24300000    26.30000000    29.59100000
H       27.96718000    26.17629000    25.28279000
H       27.68820000    24.38239000    25.34559000
H       26.56993000    26.61156000    29.98010000
H       27.32039000    24.95969000    30.19977000
--
0 1
C       32.16000000    24.06100000    28.81600000
C       32.25400000    23.97000000    25.01000000
C       32.66900000    23.70100000    26.45000000
O       33.74600000    23.16600000    26.71700000
N       31.82800000    24.10900000    27.39500000
H       30.93300000    24.51700000    27.13900000
H       33.24000000    24.03100000    28.96600000
H       31.17300000    24.10700000    24.98500000
H       31.74934000    24.96112000    29.29678000
H       31.70477000    23.16073000    29.25450000
H       32.76357000    24.88119000    24.66349000
H       32.49434000    23.11411000    24.36216000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '011ILE-014PHE-1')] = qcdb.Molecule("""
0 1
C        6.60700000    -3.99700000    21.99700000
C       10.15300000    -3.37000000    23.27900000
C        8.08100000    -4.24000000    22.31700000
O        8.60500000    -5.33100000    22.08800000
N        8.75100000    -3.23200000    22.87000000
H        6.12500000    -4.97500000    21.96500000
H        8.27600000    -2.34700000    23.01200000
H       10.74700000    -3.64000000    22.40400000
H        6.14211700    -3.40302500    22.79767000
H        6.50064600    -3.53968000    21.00224000
H       10.53565000    -2.41231400    23.66165000
H       10.27309000    -4.17774200    24.01597000
--
0 1
C        7.88900000    -8.33900000    24.05700000
C        9.43800000    -5.63700000    26.29300000
C        9.15700000    -7.00300000    25.66800000
O        9.81200000    -7.98900000    26.01000000
N        8.19800000    -7.07600000    24.74200000
H        7.63600000    -6.25400000    24.53500000
H        7.71400000    -9.11400000    24.80200000
H       10.42600000    -5.68600000    26.75200000
H        6.99501800    -8.24895900    23.42242000
H        8.73123000    -8.69944300    23.44812000
H        9.42975600    -4.82836500    25.54732000
H        8.69941400    -5.44013900    27.08403000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '012LEU-085ASP-1')] = qcdb.Molecule("""
0 1
C       12.00300000     7.41700000     2.33500000
C       12.21000000     3.98800000     0.65700000
C       11.69600000     5.96800000     1.99600000
O       10.80700000     5.37900000     2.61200000
N       12.39600000     5.38800000     1.02300000
H       12.14400000     7.43500000     3.41600000
H       13.08200000     5.93800000     0.52300000
H       11.83700000     3.44500000     1.52600000
H       12.89158000     7.82131400     1.82809900
H       11.13620000     8.05029000     2.09498600
H       11.49247000     3.90083100    -0.17219030
H       13.20110000     3.59431700     0.38729420
--
0 1
C        8.25300000     5.74700000     5.50500000
C       10.05100000     2.62600000     4.39500000
C        9.22000000     3.53900000     5.27200000
O        8.58800000     3.10900000     6.24200000
N        9.15200000     4.80000000     4.87400000
H        9.67500000     5.08600000     4.05200000
H        8.49500000     5.82400000     6.56600000
H       10.55200000     3.23300000     3.64500000
H        8.38614300     6.72893200     5.02742100
H        7.21057300     5.41751000     5.38341900
H       10.79572000     2.06500400     4.97867700
H        9.39303400     1.92446000     3.86121900
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '014LYS-018GLU-1')] = qcdb.Molecule("""
0 1
C       16.22000000    16.91300000    11.79900000
C       13.19600000    14.75900000    12.64600000
C       15.38800000    15.68000000    12.13200000
O       15.91800000    14.56800000    12.19500000
N       14.08300000    15.86700000    12.31500000
H       17.16900000    16.55200000    11.39500000
H       13.71300000    16.81100000    12.28100000
H       13.61500000    14.21200000    13.49400000
H       15.71695000    17.53278000    11.04216000
H       16.42626000    17.49011000    12.71246000
H       12.20434000    15.13931000    12.93231000
H       13.12258000    14.03341000    11.82251000
--
0 1
C       16.14000000    10.66600000    12.71800000
C       16.84400000    12.38900000     9.37400000
C       16.87000000    11.29200000    10.43600000
O       17.36600000    10.19000000    10.19800000
N       16.30600000    11.60200000    11.60300000
H       15.98200000    12.55500000    11.73500000
H       17.12100000    10.35800000    13.08500000
H       16.92200000    11.90300000     8.41300000
H       15.61356000    11.20605000    13.51876000
H       15.58750000     9.76263000    12.42023000
H       15.88699000    12.93126000     9.36500000
H       17.68895000    13.07583000     9.52996800
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '014VAL-017GLU-1')] = qcdb.Molecule("""
0 1
C       10.38200000    95.07800000    19.05400000
C        8.57800000    91.68400000    19.21700000
C        8.78800000    93.16800000    18.93600000
O        7.87500000    93.82900000    18.40500000
N        9.98900000    93.67000000    19.26700000
H       10.67400000    93.05100000    19.69900000
H        9.65700000    95.48200000    18.34400000
H        9.47700000    91.31100000    19.70800000
H       10.33263000    95.69007000    19.96665000
H       11.40499000    95.05660000    18.65023000
H        7.70540900    91.52917000    19.86862000
H        8.44270900    91.12988000    18.27644000
--
0 1
C       13.71000000    90.61200000    18.69700000
C       11.31900000    89.13100000    21.24900000
C       12.56400000    90.50900000    19.69300000
O       11.86900000    91.48700000    19.96800000
N       12.37100000    89.32200000    20.25300000
H       13.86200000    89.61600000    18.27900000
H       12.96900000    88.52900000    20.01900000
H       10.45500000    89.73900000    20.98700000
H       13.50555000    91.34077000    17.89882000
H       14.62401000    90.92047000    19.22560000
H       11.01579000    88.07482000    21.29953000
H       11.63536000    89.40124000    22.26727000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '015GLN-018LEU-1')] = qcdb.Molecule("""
0 1
C        1.80300000     4.53500000    38.27400000
C       -1.68700000     5.80100000    37.42600000
C        0.52100000     4.83600000    37.53700000
O        0.44900000     4.49100000    36.35200000
N       -0.44700000     5.49800000    38.15700000
H        2.58300000     4.47900000    37.51100000
H       -0.37100000     5.70900000    39.14900000
H       -2.00700000     4.88500000    36.94100000
H        2.05480900     5.34169200    38.97816000
H        1.76967900     3.57664900    38.81294000
H       -2.44452800     6.17223900    38.13193000
H       -1.48605000     6.52935500    36.62655000
--
0 1
C        2.02300000     6.20200000    33.62800000
C       -0.13900000     8.69500000    35.49200000
C        0.50300000     7.92500000    34.35300000
O        0.20900000     8.18000000    33.18000000
N        1.39500000     6.99100000    34.68200000
H        1.65600000     6.85500000    35.65300000
H        2.47200000     6.89800000    32.92200000
H       -1.01700000     9.21100000    35.09700000
H        2.81022800     5.56530300    34.05799000
H        1.26451300     5.61665300    33.08757000
H       -0.42638430     7.99309000    36.28870000
H        0.57226140     9.43715500    35.88355000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '015LEU-026LEU-1')] = qcdb.Molecule("""
0 1
C       39.68500000     9.51600000     9.12800000
C       36.88800000    12.06900000     8.71200000
C       37.76100000    10.85200000     8.43800000
O       37.42800000    10.03800000     7.57400000
N       38.85000000    10.70600000     9.18500000
H       39.06900000    11.42500000     9.86700000
H       39.66700000     9.09800000     8.12100000
H       37.33900000    12.65900000     9.50900000
H       40.72381000     9.76042500     9.39471100
H       39.32454000     8.72053000     9.79680300
H       36.78938000    12.68195000     7.80394700
H       35.89571000    11.71329000     9.02636800
--
0 1
C       39.89400000    12.81400000    13.62800000
C       39.65000000    15.12100000    10.62300000
C       39.63600000    13.23400000    12.18600000
O       39.13800000    12.44900000    11.37300000
N       39.97100000    14.48000000    11.88800000
H       40.31600000    13.66100000    14.17300000
H       40.36900000    15.06400000    12.62000000
H       39.08400000    14.44300000     9.98800000
H       38.93213000    12.54785000    14.09057000
H       40.60493000    11.97693000    13.69035000
H       39.04096000    16.01985000    10.79947000
H       40.58788000    15.39983000    10.12038000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '017ASN-021LYS-2')] = qcdb.Molecule("""
0 1
C       18.31000000    53.33500000    13.55200000
C       15.15800000    51.25200000    13.89900000
C       17.40800000    52.16400000    13.91800000
O       17.89100000    51.12400000    14.35300000
N       16.11200000    52.31800000    13.67200000
H       19.32100000    52.93600000    13.43500000
H       15.77000000    53.22700000    13.37500000
H       15.29600000    50.87600000    14.91300000
H       18.00289000    53.79188000    12.59966000
H       18.32156000    54.09524000    14.34692000
H       14.12871000    51.63311000    13.82610000
H       15.32087000    50.40273000    13.21914000
--
0 1
C       18.14100000    47.14800000    15.07000000
C       19.40300000    48.50500000    11.74600000
C       19.20000000    47.57500000    12.93200000
O       19.69100000    46.45100000    12.94400000
N       18.42200000    48.01000000    13.91900000
H       18.05200000    48.95500000    13.89000000
H       19.08200000    46.77900000    15.48700000
H       19.70200000    47.88000000    10.90300000
H       17.61988000    47.71872000    15.85276000
H       17.56658000    46.26118000    14.76408000
H       18.45753000    49.00950000    11.49786000
H       20.20791000    49.23261000    11.92684000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '017LEU-025ILE-1')] = qcdb.Molecule("""
0 1
C        7.17000000    24.28800000     3.37300000
C        4.69000000    27.02900000     2.57900000
C        5.88700000    25.09500000     3.42500000
O        4.98800000    24.82000000     4.21500000
N        5.80200000    26.09500000     2.57000000
H        7.14200000    23.76600000     2.41800000
H        6.52900000    26.20600000     1.87300000
H        3.86600000    26.65200000     3.18100000
H        8.05329200    24.94218000     3.41595900
H        7.20100600    23.54889000     4.18710000
H        5.00782900    27.99179000     3.00563500
H        4.32016700    27.16506000     1.55200900
--
0 1
C        2.15600000    22.62800000     3.98800000
C        1.72200000    26.09700000     5.49600000
C        1.39900000    24.65900000     5.10600000
O        0.33200000    24.13600000     5.44200000
N        2.32800000    24.01700000     4.40300000
H        3.20500000    24.48400000     4.20000000
H        1.13100000    22.45900000     3.65100000
H        2.57900000    26.42800000     4.91000000
H        2.84211400    22.38362000     3.16366600
H        2.34216800    21.96301000     4.84423000
H        0.86013660    26.74641000     5.28279400
H        2.01418400    26.14415000     6.55543600
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '019LEU-022MET-1')] = qcdb.Molecule("""
0 1
C       15.07200000     3.59400000     7.20300000
C       18.65000000     4.93300000     7.58500000
C       16.44300000     3.91200000     7.78300000
O       16.73400000     3.48300000     8.90700000
N       17.30400000     4.65300000     7.06700000
H       14.42400000     3.43700000     8.06600000
H       17.04400000     4.96100000     6.13800000
H       19.11400000     3.97900000     7.84500000
H       14.65841000     4.41967300     6.60533400
H       15.10208000     2.66939900     6.60784600
H       19.27722000     5.41750600     6.82220700
H       18.61226000     5.54113500     8.50083200
--
0 1
C       15.95900000     4.70400000    12.62200000
C       17.33800000     7.33700000    10.22600000
C       16.99400000     6.62000000    11.52800000
O       17.01400000     7.26900000    12.58600000
N       16.59900000     5.34800000    11.47400000
H       16.61400000     4.85500000    10.58600000
H       15.93900000     5.37200000    13.48600000
H       18.28000000     7.86400000    10.38200000
H       14.91518000     4.44913000    12.38646000
H       16.50130000     3.79414900    12.91878000
H       17.43170000     6.66521900     9.36001500
H       16.55190000     8.08422700    10.04243000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '023LYS-146SER-2')] = qcdb.Molecule("""
0 1
C        0.57400000   -11.81000000     5.51200000
C        2.98900000   -11.59600000     8.42900000
C        1.66800000   -11.55200000     7.67900000
O        0.59700000   -11.48100000     8.28100000
N        1.74900000   -11.62500000     6.35700000
H        2.66800000   -11.56900000     5.92300000
H       -0.29900000   -12.08100000     6.10800000
H        3.79700000   -11.38500000     7.73300000
H        0.34273930   -10.88784000     4.95870300
H        0.77769330   -12.62477000     4.80161000
H        2.99807600   -10.85854000     9.24512700
H        3.16340300   -12.60725000     8.82517500
--
0 1
C        5.24800000   -11.18400000     3.29500000
C        6.86400000    -9.73600000     6.40300000
C        5.38400000   -10.70700000     4.72800000
O        4.46400000   -10.86500000     5.52700000
N        6.52500000   -10.10900000     5.04000000
H        6.24800000   -11.24000000     2.86400000
H        7.25300000   -10.02600000     4.34000000
H        6.01600000    -9.92500000     7.03800000
H        4.76020600   -12.16937000     3.26187800
H        4.67695500   -10.44496000     2.71386000
H        7.67955300   -10.37048000     6.78023800
H        7.13044300    -8.67023000     6.45905500
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '024PHE-028ALA-1')] = qcdb.Molecule("""
0 1
C       13.10800000    32.54900000    28.14000000
C       12.74900000    35.41500000    25.67100000
C       12.61200000    33.37600000    26.94100000
O       11.68000000    32.95400000    26.24100000
N       13.25400000    34.52700000    26.71000000
H       13.63300000    33.17300000    28.85800000
H       13.97900000    34.85100000    27.34800000
H       11.78900000    35.82100000    25.99100000
H       12.21265000    32.12384000    28.61706000
H       13.75589000    31.72402000    27.80886000
H       13.45014000    36.25053000    25.52858000
H       12.57856000    34.88148000    24.72427000
--
0 1
C        8.18500000    33.56600000    24.48600000
C       10.66300000    30.69400000    23.84300000
C        9.37100000    31.51800000    23.84600000
O        8.40500000    31.23100000    23.11500000
N        9.36100000    32.65500000    24.56900000
H       10.15900000    32.90600000    25.14600000
H        7.28900000    33.04600000    24.83300000
H       10.61500000    30.03600000    22.97200000
H        8.36338800    34.44931000    25.11683000
H        8.02420200    33.89262000    23.44799000
H       11.53755000    31.35534000    23.75472000
H       10.73578000    30.06750000    24.74422000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '024THR-041GLU-2')] = qcdb.Molecule("""
0 1
C        1.19400000    22.61400000     8.20600000
C        3.82700000    20.42100000     9.80300000
C        2.28100000    21.57600000     8.29300000
O        2.65400000    20.99200000     7.30100000
N        2.78900000    21.39200000     9.48900000
H        1.21100000    23.22600000     9.10700000
H        2.41600000    21.94700000    10.25500000
H        4.18100000    19.95300000     8.89000000
H        1.38331300    23.25161000     7.32986100
H        0.20395010    22.13755000     8.15314000
H        4.67854600    20.86907000    10.33601000
H        3.34354000    19.68787000    10.46541000
--
0 1
C        2.44000000    18.11300000     4.50400000
C        5.77900000    19.46400000     5.76700000
C        4.73000000    18.81000000     4.86600000
O        5.06900000    18.22100000     3.82800000
N        3.44900000    18.88600000     5.22500000
H        3.19000000    19.39500000     6.06700000
H        2.64600000    17.04700000     4.62100000
H        5.29700000    20.08800000     6.52100000
H        1.44433900    18.32759000     4.91946100
H        2.44285400    18.33914000     3.42750100
H        6.31251500    18.62228000     6.23269500
H        6.45927300    20.07380000     5.15432300
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '025GLY-029SER-2')] = qcdb.Molecule("""
0 1
C        7.13200000    67.30800000    -0.96500000
C       10.74700000    68.19700000    -0.17500000
C        8.63500000    67.16300000    -0.80700000
O        9.18300000    66.07800000    -1.00400000
N        9.30600000    68.25100000    -0.43700000
H        6.67100000    66.32300000    -0.88400000
H        6.93600000    67.69600000    -1.96500000
H        8.81500000    69.13900000    -0.37500000
H       11.25300000    67.75600000    -1.03600000
H        6.68160500    67.99151000    -0.23018490
H       11.14583000    69.21118000    -0.02543690
H       10.93559000    67.53729000     0.68477580
--
0 1
C       11.47700000    62.99700000    -0.75400000
C        8.62100000    63.50200000     1.75700000
C        9.68600000    62.81600000     0.91100000
O        9.91400000    61.62300000     1.08200000
N       10.39200000    63.56600000     0.05400000
H       10.10600000    64.52400000    -0.12400000
H       11.09600000    62.13800000    -1.30800000
H        8.51800000    62.91900000     2.66800000
H       11.85387000    63.73133000    -1.48113600
H       12.30145000    62.64109000    -0.11869770
H        8.90534800    64.52731000     2.03608300
H        7.65143200    63.48974000     1.23758900
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '025ILE-070TRP-1')] = qcdb.Molecule("""
0 1
C        6.01900000   -13.17600000    -7.12800000
C        4.25900000   -14.74800000    -4.14500000
C        5.53000000   -14.25700000    -6.16900000
O        5.68200000   -15.44700000    -6.45200000
N        4.95200000   -13.83800000    -5.04500000
H        5.96200000   -12.20800000    -6.62800000
H        4.83400000   -12.84400000    -4.88600000
H        4.82600000   -15.66800000    -3.99700000
H        7.05334900   -13.39867000    -7.42890100
H        5.36553800   -13.16252000    -8.01276300
H        4.07119100   -14.25464000    -3.17994900
H        3.26892500   -14.96352000    -4.57314000
--
0 1
C        6.66100000   -17.72800000    -9.49500000
C        7.06900000   -18.48300000    -5.78500000
C        6.92400000   -18.78600000    -7.26900000
O        7.08500000   -19.93800000    -7.69900000
N        6.63300000   -17.73500000    -8.03700000
H        6.44800000   -16.84900000    -7.57700000
H        6.98900000   -18.69800000    -9.86000000
H        6.58900000   -17.52900000    -5.60200000
H        5.65266600   -17.51183000    -9.87779400
H        7.37585900   -16.95810000    -9.82092300
H        6.58467900   -19.26304000    -5.17922200
H        8.13849500   -18.40056000    -5.54130900
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '027VAL-031LEU-1')] = qcdb.Molecule("""
0 1
C       10.66300000    30.69400000    23.84300000
C        8.18500000    33.56600000    24.48600000
C        9.37100000    31.51800000    23.84600000
O        8.40500000    31.23100000    23.11500000
N        9.36100000    32.65500000    24.56900000
H       10.61500000    30.03600000    22.97200000
H       10.15900000    32.90600000    25.14600000
H        7.28900000    33.04600000    24.83300000
H       11.53755000    31.35534000    23.75472000
H       10.73578000    30.06750000    24.74422000
H        8.36338800    34.44931000    25.11683000
H        8.02420200    33.89262000    23.44799000
--
0 1
C        4.81000000    31.03100000    21.67200000
C        7.95800000    31.28100000    19.57600000
C        6.48000000    31.14800000    19.92900000
O        5.62100000    31.06800000    19.03100000
N        6.19900000    31.16400000    21.23000000
H        6.95100000    31.23600000    21.91100000
H        4.38000000    30.11100000    21.27900000
H        8.03200000    31.47500000    18.50400000
H        4.75122500    31.05824000    22.77009000
H        4.19712700    31.85536000    21.27852000
H        8.43335500    32.10899000    20.12233000
H        8.46768900    30.33135000    19.79598000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '027VAL-068GLU-2')] = qcdb.Molecule("""
0 1
C        1.24900000   -16.76400000    -5.34400000
C        4.25900000   -14.74800000    -4.14500000
C        2.89000000   -15.04600000    -4.73700000
O        2.16200000   -14.11100000    -5.10300000
N        2.51800000   -16.32300000    -4.77000000
H        3.20200000   -17.03400000    -4.52000000
H        0.57800000   -15.91500000    -5.48500000
H        4.82600000   -15.66800000    -3.99700000
H        1.41459600   -17.23520000    -6.32407400
H        0.77053450   -17.47086000    -4.65015300
H        4.78275400   -14.06024000    -4.82520000
H        4.07119100   -14.25464000    -3.17994900
--
0 1
C        4.49800000   -20.56900000    -3.98500000
C        7.06900000   -18.48300000    -5.78500000
C        5.18700000   -19.34500000    -4.54600000
O        4.58900000   -18.26300000    -4.62200000
N        6.42700000   -19.51700000    -4.98200000
H        5.26800000   -21.09500000    -3.41800000
H        6.85900000   -20.43400000    -4.92500000
H        6.58900000   -17.52900000    -5.60200000
H        3.70499100   -20.25497000    -3.29036000
H        4.15245500   -21.27809000    -4.75167800
H        8.13849500   -18.40056000    -5.54130900
H        6.96417200   -18.70205000    -6.85785900
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '029GLU-032VAL-1')] = qcdb.Molecule("""
0 1
C        5.68300000     7.08200000     2.74100000
C        2.70200000     8.99900000     1.32500000
C        4.86500000     8.21300000     2.12400000
O        5.38800000     9.30900000     1.92800000
N        3.58100000     7.97000000     1.86200000
H        6.49700000     7.55500000     3.29400000
H        3.21700000     7.03400000     2.00200000
H        3.17500000     9.43700000     0.44400000
H        5.07648700     6.49132900     3.44331800
H        6.12600800     6.44481700     1.96142200
H        1.73981700     8.56334900     1.01773600
H        2.55311400     9.80169100     2.06223800
--
0 1
C        6.04200000    11.71500000     4.59100000
C        2.33600000    10.75500000     4.70200000
C        3.59600000    11.62200000     4.73100000
O        3.50700000    12.85200000     4.80600000
N        4.76800000    10.99100000     4.63400000
H        4.77300000     9.97500000     4.59500000
H        6.09800000    12.36300000     5.46600000
H        1.49600000    11.41900000     4.49500000
H        6.90442800    11.03316000     4.62726000
H        6.09301700    12.37176000     3.71005400
H        2.40736900    10.00750000     3.89816100
H        2.15798000    10.27384000     5.67503500
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '030TRP-178GLY-2')] = qcdb.Molecule("""
0 1
C       48.55800000    20.43700000    27.72000000
C       49.42900000    23.30200000    25.41700000
C       49.29300000    22.44600000    26.64900000
O       49.66600000    22.89100000    27.73100000
N       48.73300000    21.25500000    26.53200000
H       48.43900000    20.89900000    25.62600000
H       49.13700000    20.85700000    28.53100000
H       49.02700000    22.76300000    24.55500000
H       47.50412000    20.44059000    28.03516000
H       48.94433000    19.41797000    27.57057000
H       48.86150000    24.22684000    25.59757000
H       50.49582000    23.50705000    25.24422000
--
0 1
C       46.91000000    18.45700000    22.92300000
C       46.99900000    22.00400000    21.65200000
C       47.26200000    19.92500000    22.86800000
O       47.97300000    20.45000000    23.72500000
N       46.74800000    20.58800000    21.84400000
H       47.35100000    17.96200000    22.05600000
H       45.82600000    18.39600000    22.83300000
H       46.17900000    20.09300000    21.16600000
H       47.24700000    22.46000000    22.60400000
H       47.23510000    17.94775000    23.84223000
H       47.83769000    22.16152000    20.95789000
H       46.11238000    22.51262000    21.24554000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '033ASN-036TRP-2')] = qcdb.Molecule("""
0 1
C      -15.84200000   -16.31900000    11.52700000
C      -14.66100000   -17.23500000     8.03600000
C      -15.06100000   -16.27300000    10.22700000
O      -14.11900000   -15.49600000    10.09500000
N      -15.43600000   -17.11400000     9.26800000
H      -16.62900000   -17.07200000    11.44700000
H      -16.19400000   -17.76200000     9.44200000
H      -13.61300000   -17.40400000     8.29400000
H      -15.14067000   -16.58153000    12.33274000
H      -16.33130000   -15.35410000    11.72589000
H      -15.01129000   -18.09848000     7.45145600
H      -14.69713000   -16.32606000     7.41751600
--
0 1
C      -14.81600000   -11.35000000     9.08200000
C      -11.55600000   -11.57000000    11.05000000
C      -13.82900000   -11.74800000    10.16200000
O      -14.18600000   -12.49300000    11.07300000
N      -12.59200000   -11.26400000    10.06400000
H      -14.31300000   -10.68000000     8.38500000
H      -12.36700000   -10.60800000     9.32200000
H      -11.85500000   -12.42300000    11.66100000
H      -15.16270000   -12.26473000     8.57893800
H      -15.66921000   -10.82007000     9.53056500
H      -10.60806000   -11.83272000    10.55767000
H      -11.39262000   -10.72477000    11.73475000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '033VAL-115VAL-1')] = qcdb.Molecule("""
0 1
C        3.50000000   -15.42200000    -8.51000000
C        6.13100000   -14.90800000    -5.84500000
C        4.74700000   -15.67800000    -7.69300000
O        5.48800000   -16.63000000    -7.94400000
N        4.98700000   -14.79900000    -6.73400000
H        2.98500000   -14.54900000    -8.11300000
H        4.29600000   -14.08000000    -6.53900000
H        6.82200000   -15.64400000    -6.24400000
H        3.82476300   -15.22699000    -9.54271500
H        2.79753700   -16.26812000    -8.48486100
H        6.68485000   -13.96160000    -5.75792600
H        5.79784000   -15.27092000    -4.86148900
--
0 1
C        6.70200000   -18.82500000   -10.87800000
C        8.96800000   -16.86100000    -8.52700000
C        8.55800000   -17.77100000    -9.67700000
O        9.41400000   -18.41500000   -10.29300000
N        7.26400000   -17.85600000    -9.94300000
H        6.61100000   -17.32000000    -9.37700000
H        7.48300000   -19.33800000   -11.44000000
H        8.07700000   -16.42400000    -8.07400000
H        6.11234300   -19.56737000   -10.32016000
H        6.04861200   -18.28619000   -11.57997000
H        9.49428800   -17.49996000    -7.80259900
H        9.62625800   -16.05438000    -8.88205600
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '035TRP-056ILE-1')] = qcdb.Molecule("""
0 1
C       26.94200000     3.32900000    60.85700000
C       23.36400000     4.46100000    60.67500000
C       25.64000000     3.92200000    61.37000000
O       25.42900000     4.11100000    62.57300000
N       24.76500000     4.19500000    60.41200000
H       27.22000000     3.86700000    59.95300000
H       25.02900000     3.97900000    59.45500000
H       23.17300000     4.49000000    61.74800000
H       27.73424000     3.40537800    61.61629000
H       26.76701000     2.27547500    60.59344000
H       23.07798000     5.43057000    60.24126000
H       22.78204000     3.63957300    60.23163000
--
0 1
C       25.57300000     1.32400000    65.13200000
C       23.05800000     4.13700000    65.27600000
C       23.71400000     2.82900000    65.62100000
O       23.23400000     2.07500000    66.47400000
N       24.83200000     2.56700000    64.96700000
H       25.16900000     3.22600000    64.27100000
H       25.22900000     0.77300000    66.00000000
H       23.34500000     4.41800000    64.26200000
H       25.44655000     0.69534950    64.23824000
H       26.64101000     1.56951800    65.22732000
H       21.96725000     4.01821300    65.35444000
H       23.48034000     4.88360800    65.96463000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '037PRO-041ALA-1')] = qcdb.Molecule("""
0 1
C        1.45400000     4.71900000    34.53000000
C        3.13000000     7.08600000    32.05200000
C        1.84000000     5.40400000    33.22100000
O        1.32400000     5.02600000    32.16500000
N        2.69900000     6.41100000    33.26800000
H        1.21000000     3.67600000    34.32800000
H        3.04100000     6.73500000    34.16200000
H        2.25900000     7.29900000    31.42900000
H        2.25703800     4.78781000    35.27860000
H        0.59452870     5.20371900    35.01617000
H        3.61697200     8.04335100    32.28935000
H        3.79117200     6.41541400    31.48352000
--
0 1
C        1.18400000     4.22900000    28.45700000
C        3.04300000     1.89200000    30.88400000
C        2.30300000     2.39200000    29.64800000
O        2.12700000     1.63200000    28.69600000
N        1.86300000     3.66100000    29.62200000
H        1.95300000     4.23600000    30.45200000
H        0.36500000     3.56500000    28.17100000
H        3.62500000     1.01900000    30.57900000
H        0.75336240     5.20646700    28.71989000
H        1.86135100     4.31048600    27.59412000
H        3.71345000     2.67305600    31.27188000
H        2.30265300     1.59170400    31.64011000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '038ALA-041GLY-1')] = qcdb.Molecule("""
0 1
C       17.65300000     1.60500000    -0.70200000
C       15.39900000    -1.42800000    -0.29600000
C       16.38900000     0.75300000    -0.62700000
O       15.27600000     1.25100000    -0.74500000
N       16.54900000    -0.54500000    -0.39900000
H       17.36000000     2.64900000    -0.58200000
H       17.48800000    -0.93600000    -0.36800000
H       14.79900000    -1.35800000    -1.20500000
H       18.36375000     1.32388100     0.08907477
H       18.14824000     1.48370200    -1.67669100
H       15.74769000    -2.46461400    -0.17831850
H       14.74806000    -1.12601200     0.53771800
--
0 1
C       13.83700000     3.27200000     2.04300000
C       14.33400000    -0.32700000     3.19500000
C       13.71500000     1.04600000     3.00400000
O       12.57400000     1.27300000     3.44000000
N       14.41100000     1.96700000     2.33500000
H       15.37200000     1.77400000     2.06400000
H       13.55000000     3.77000000     2.97000000
H       14.59000000     3.85800000     1.55800000
H       13.50500000    -1.02800000     3.31800000
H       12.95501000     3.19940900     1.38967700
H       14.91210000    -0.61979280     2.30613800
H       15.01510000    -0.47284690     4.04637200
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '038GLU-047GLU-2')] = qcdb.Molecule("""
0 1
C      -16.11100000    27.50200000   -23.68000000
C      -19.81800000    28.30500000   -23.42700000
C      -17.57900000    27.62700000   -24.05100000
O      -17.97700000    27.26800000   -25.14600000
N      -18.39800000    28.13100000   -23.14300000
H      -16.07300000    27.42500000   -22.59400000
H      -18.03600000    28.47400000   -22.25800000
H      -20.06200000    28.01900000   -24.45000000
H      -15.66274000    26.61151000   -24.14486000
H      -15.56511000    28.41564000   -23.95797000
H      -20.44833000    27.71065000   -22.74918000
H      -20.08475000    29.36545000   -23.30747000
--
0 1
C      -17.09400000    26.75600000   -28.46700000
C      -19.69700000    24.69900000   -26.61100000
C      -18.90800000    25.25800000   -27.78600000
O      -19.13600000    24.89500000   -28.94000000
N      -17.96900000    26.13900000   -27.48500000
H      -17.81700000    26.37700000   -26.52100000
H      -17.21900000    26.27900000   -29.44000000
H      -19.53500000    25.34800000   -25.75200000
H      -17.31758000    27.82730000   -28.57807000
H      -16.04417000    26.62794000   -28.16457000
H      -20.77290000    24.68317000   -26.83945000
H      -19.31530000    23.70215000   -26.34532000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '039SER-046MET-1')] = qcdb.Molecule("""
0 1
C       37.13200000    14.35600000    -9.09000000
C       37.95400000    10.90500000   -10.49300000
C       36.97200000    13.07700000    -9.89500000
O       35.97000000    12.91200000   -10.57200000
N       37.96700000    12.20500000    -9.82700000
H       38.12900000    14.38400000    -8.64600000
H       38.74600000    12.40400000    -9.21500000
H       36.96600000    10.70700000   -10.89800000
H       36.95649000    15.24861000    -9.70842300
H       36.39719000    14.34813000    -8.27147300
H       38.70130000    10.84719000   -11.29811000
H       38.20371000    10.13707000    -9.74605600
--
0 1
C       32.75800000    11.46200000   -10.10200000
C       34.57500000    10.81700000   -13.39000000
C       33.45800000    10.81100000   -12.34800000
O       32.36000000    10.30800000   -12.60000000
N       33.73400000    11.39100000   -11.18300000
H       34.65700000    11.78700000   -11.03800000
H       32.02300000    10.66300000   -10.19700000
H       35.46500000    11.27400000   -12.95300000
H       33.26581000    11.34828000    -9.13287600
H       32.21483000    12.41853000   -10.10563000
H       34.81162000     9.78669800   -13.69412000
H       34.25590000    11.43015000   -14.24570000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '040THR-048VAL-2')] = qcdb.Molecule("""
0 1
C        7.24400000    -6.35700000     9.43600000
C        8.35500000    -4.20300000    12.36200000
C        7.42800000    -5.90300000    10.88200000
O        6.82800000    -6.48300000    11.80000000
N        8.22400000    -4.85900000    11.07700000
H        7.92400000    -5.79500000     8.79500000
H        8.75800000    -4.49000000    10.29500000
H        7.63700000    -4.61500000    13.06400000
H        7.43872200    -7.43736800     9.36608100
H        6.21398500    -6.12580100     9.12678100
H        9.36929100    -4.33457600    12.76685000
H        8.15464700    -3.12719300    12.25021000
--
0 1
C        3.78500000    -6.44400000    13.96100000
C        7.27600000    -6.83600000    15.43900000
C        5.75700000    -6.85100000    15.34700000
O        5.07200000    -7.13400000    16.32800000
N        5.23100000    -6.59400000    14.15400000
H        5.86000000    -6.39800000    13.38200000
H        3.27500000    -6.67300000    14.89800000
H        7.68900000    -6.46100000    14.49800000
H        3.49436200    -5.41141400    13.71749000
H        3.33258800    -7.14564000    13.24474000
H        7.57248100    -6.17648200    16.26794000
H        7.65234400    -7.85801800    15.59342000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '040THR-049GLN-2')] = qcdb.Molecule("""
0 1
C        7.24400000    -6.35700000     9.43600000
C        8.61000000    -9.79500000     8.59000000
C        8.47400000    -8.29300000     8.59900000
O        9.25100000    -7.59500000     7.94800000
N        7.50300000    -7.79400000     9.34300000
H        6.90800000    -8.42600000     9.87200000
H        7.92400000    -5.79500000     8.79500000
H        7.76400000   -10.23900000     9.11900000
H        6.21398500    -6.12580100     9.12678100
H        7.37657100    -6.02989500    10.47784000
H        8.64695400   -10.12382000     7.54094700
H        9.54550000   -10.07275000     9.09763600
--
0 1
C        3.46200000    -9.09800000    11.28000000
C        5.20700000   -12.30500000    10.21500000
C        4.60200000   -10.05000000    10.92500000
O        5.76500000    -9.63700000    10.81600000
N        4.27000000   -11.31900000    10.73100000
H        2.67700000    -9.66500000    11.78600000
H        3.31500000   -11.61700000    10.88600000
H        5.92400000   -11.79500000     9.57100000
H        3.85491800    -8.33492000    11.96799000
H        3.03655100    -8.60627500    10.39276000
H        5.77137800   -12.82757000    11.00138000
H        4.65710300   -13.03771000     9.60610800
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '041GLY-045LEU-2')] = qcdb.Molecule("""
0 1
C       13.83700000     3.27200000     2.04300000
C       11.50000000     2.16700000    -0.76900000
C       12.62200000     3.17200000     1.14300000
O       11.61700000     3.85600000     1.36000000
N       12.67500000     2.32900000     0.10300000
H       13.55000000     3.77000000     2.97000000
H       14.59000000     3.85800000     1.55800000
H       13.53700000     1.81900000    -0.09100000
H       11.16200000     3.15200000    -1.10700000
H       14.27088000     2.28557400     2.26371700
H       11.72368000     1.55384000    -1.65443800
H       10.66302000     1.74269900    -0.19503180
--
0 1
C        8.04700000     5.34400000     1.30800000
C        9.26600000     3.02800000     4.07400000
C        8.37800000     4.00900000     3.30800000
O        7.28400000     4.35600000     3.74500000
N        8.84600000     4.45000000     2.14200000
H        9.77500000     4.17200000     1.83800000
H        7.77900000     6.22500000     1.89300000
H        8.64000000     2.57400000     4.84500000
H        8.62430600     5.66496500     0.42839820
H        7.09836000     4.87435500     1.00880800
H        9.63134800     2.22771300     3.41365000
H       10.10993000     3.52005200     4.57964100
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '041LEU-045LEU-1')] = qcdb.Molecule("""
0 1
C       36.48200000    37.02700000     7.19500000
C       32.85400000    36.41300000     8.20500000
C       35.01500000    36.60900000     7.07000000
O       34.63600000    35.94700000     6.10200000
N       34.20500000    36.96500000     8.06700000
H       36.88100000    37.13600000     6.18500000
H       34.60600000    37.44000000     8.86500000
H       32.96600000    35.34500000     8.38900000
H       36.56172000    37.99564000     7.71015400
H       37.06618000    36.25545000     7.71791600
H       32.33727000    36.83410000     9.08002200
H       32.23854000    36.49906000     7.29736300
--
0 1
C       33.35500000    32.73500000     4.49900000
C       33.56700000    36.12000000     2.79500000
C       33.30900000    34.62700000     2.97400000
O       32.88400000    33.92500000     2.05200000
N       33.53300000    34.14500000     4.19400000
H       33.86100000    34.77300000     4.92000000
H       33.88700000    32.15100000     3.74800000
H       33.12500000    36.42700000     1.84600000
H       33.78742000    32.52023000     5.48737800
H       32.29491000    32.44398000     4.46000500
H       33.08971000    36.66644000     3.62180000
H       34.64620000    36.33014000     2.76080800
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '042LEU-047ILE-2')] = qcdb.Molecule("""
0 1
C       27.98600000    25.44300000    -1.06900000
C       27.41300000    29.03600000     0.02400000
C       27.55500000    26.63500000    -0.22400000
O       26.97700000    26.46100000     0.84900000
N       27.82400000    27.84600000    -0.70100000
H       27.25800000    24.65500000    -0.87100000
H       28.38000000    27.94300000    -1.54400000
H       27.65500000    28.92900000     1.08400000
H       27.98148000    25.67219000    -2.14484900
H       28.98297000    25.09248000    -0.76373210
H       27.93231000    29.92357000    -0.36656000
H       26.32263000    29.15515000    -0.05904148
--
0 1
C       27.12400000    24.69000000     4.01500000
C       24.61500000    27.52600000     3.59800000
C       25.41000000    26.42500000     4.28700000
O       25.25600000    26.17700000     5.48300000
N       26.25800000    25.75500000     3.51300000
H       26.29600000    25.99600000     2.52600000
H       26.68400000    24.21300000     4.89200000
H       25.26200000    28.40300000     3.54400000
H       27.25983000    23.93366000     3.22791800
H       28.07774000    25.14166000     4.32546900
H       24.32018000    27.24857000     2.57520400
H       23.72691000    27.78661000     4.19245400
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '044SER-047GLU-1')] = qcdb.Molecule("""
0 1
C       -4.41100000    -1.31100000     3.89100000
C       -2.56600000     0.77800000     1.28300000
C       -4.08700000    -0.59800000     2.56700000
O       -4.87600000    -0.61100000     1.62300000
N       -2.93400000     0.07100000     2.50000000
H       -5.14900000    -2.08900000     3.68800000
H       -2.41400000     0.23300000     3.35700000
H       -3.40000000     1.38500000     0.94800000
H       -3.53373000    -1.76463100     4.37537300
H       -4.87483900    -0.58014620     4.56975300
H       -1.66523100     1.39371600     1.42267600
H       -2.38752400     0.02507657     0.50117360
--
0 1
C       -5.30200000    -3.26000000    -0.85800000
C       -1.62000000    -2.49700000    -0.34800000
C       -2.92200000    -2.98600000    -1.01900000
O       -2.96700000    -3.33800000    -2.19900000
N       -4.00500000    -2.98100000    -0.26300000
H       -3.95000000    -2.69900000     0.70600000
H       -5.22600000    -4.10500000    -1.54700000
H       -0.97900000    -2.11000000    -1.13900000
H       -6.02596200    -3.53589800    -0.07712933
H       -5.68418700    -2.41475100    -1.44917400
H       -1.77867100    -1.68076000     0.37212200
H       -1.10755600    -3.31988200     0.17187120
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '044TRP-054LEU-1')] = qcdb.Molecule("""
0 1
C       10.95000000   -17.77800000    14.61300000
C        8.09500000   -17.57500000    17.10300000
C        9.73000000   -18.22000000    15.39800000
O        9.33700000   -19.39000000    15.34600000
N        9.15700000   -17.28500000    16.14700000
H       11.04400000   -16.70100000    14.73600000
H        9.54800000   -16.34800000    16.13300000
H        7.88900000   -18.64300000    17.12000000
H       11.83458000   -18.28799000    15.02219000
H       10.81699000   -17.96678000    13.53751000
H        8.42178300   -17.28645000    18.11293000
H        7.16770200   -17.04780000    16.83434000
--
0 1
C       10.86300000   -22.79600000    14.63400000
C        8.21700000   -22.12500000    17.25000000
C        9.09500000   -22.92200000    16.29800000
O        9.07200000   -24.15500000    16.27100000
N        9.88700000   -22.20000000    15.52000000
H        9.81900000   -21.18700000    15.55200000
H       10.78000000   -23.87900000    14.66100000
H        8.35200000   -21.07100000    17.01400000
H       10.68926000   -22.44134000    13.60734000
H       11.86068000   -22.50299000    14.99285000
H        7.15596500   -22.39611000    17.14654000
H        8.57411400   -22.27476000    18.27958000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '048GLU-052LYS-1')] = qcdb.Molecule("""
0 1
C       29.19600000    -4.52700000    -8.01800000
C       30.56800000    -5.20200000    -4.49700000
C       29.56100000    -5.28400000    -6.71900000
O       29.21100000    -6.46300000    -6.56200000
N       30.18000000    -4.58900000    -5.75600000
H       28.33500000    -5.03100000    -8.46100000
H       30.48000000    -3.64400000    -5.95100000
H       31.12100000    -6.11400000    -4.73200000
H       28.93132000    -3.49150700    -7.75781500
H       30.02741000    -4.54496500    -8.73802700
H       31.25723000    -4.56627900    -3.92182400
H       29.69681000    -5.52679000    -3.90918000
--
0 1
C       28.28100000    -9.83500000    -5.01600000
C       25.58300000    -7.16200000    -5.57000000
C       26.25400000    -8.54400000    -5.36800000
O       25.56900000    -9.57900000    -5.29400000
N       27.58200000    -8.56000000    -5.20700000
H       28.11800000    -7.70900000    -5.35700000
H       27.81800000   -10.58500000    -5.66300000
H       24.54500000    -7.26700000    -5.25000000
H       29.34249000    -9.78675100    -5.30045600
H       28.17246000   -10.21160000    -3.98818900
H       26.06876000    -6.39287400    -4.95155100
H       25.58721000    -6.87263200    -6.63124900
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '051ALA-054VAL-1')] = qcdb.Molecule("""
0 1
C       11.19000000    25.57800000    16.41500000
C       11.03900000    29.37100000    16.32400000
C       10.62200000    26.98500000    16.41900000
O        9.40700000    27.16400000    16.50900000
N       11.48500000    27.98400000    16.29700000
H       10.43800000    24.94100000    15.95400000
H       12.47400000    27.79300000    16.18300000
H       10.24500000    29.45700000    17.07000000
H       12.10912000    25.48662000    15.81761000
H       11.34264000    25.22586000    17.44587000
H       11.84798000    30.03823000    16.65618000
H       10.58828000    29.70976000    15.37949000
--
0 1
C        6.39400000    27.35400000    13.71200000
C        3.59400000    27.50000000    16.23200000
C        5.69100000    27.63400000    15.02000000
O        6.30800000    28.04200000    16.02200000
N        4.39500000    27.37700000    15.02700000
H        5.79400000    27.77400000    12.90900000
H        3.92500000    27.14300000    14.15800000
H        4.23000000    27.46900000    17.12000000
H        7.38353000    27.83373000    13.73832000
H        6.45338400    26.26317000    13.58333000
H        3.05653700    28.45976000    16.23055000
H        2.88687400    26.65965000    16.29349000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '051ALA-055ASN-1')] = qcdb.Molecule("""
0 1
C       15.43500000    31.69400000    28.98700000
C       18.02000000    32.43800000    26.28000000
C       16.54400000    32.41200000    28.22200000
O       17.05600000    33.43400000    28.66700000
N       16.94500000    31.87500000    27.07200000
H       15.02000000    32.40600000    29.70300000
H       16.49700000    31.01700000    26.75900000
H       18.89500000    32.58600000    26.91400000
H       14.63110000    31.37813000    28.30585000
H       15.83188000    30.83962000    29.55490000
H       18.28187000    31.71785000    25.49082000
H       17.73736000    33.42688000    25.88983000
--
0 1
C       18.69200000    36.91200000    28.75800000
C       14.96700000    36.73500000    27.93400000
C       16.29200000    37.30600000    28.41700000
O       16.37800000    38.47000000    28.76200000
N       17.35400000    36.50400000    28.36800000
H       17.21000000    35.53100000    28.11800000
H       18.64700000    37.38300000    29.74400000
H       14.36000000    37.58000000    27.59900000
H       19.32477000    36.01379000    28.81109000
H       19.15022000    37.62747000    28.05934000
H       15.13268000    36.04503000    27.09347000
H       14.44896000    36.21332000    28.75222000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '052CYS-056ALA-1')] = qcdb.Molecule("""
0 1
C       26.72000000    17.60100000    -3.75600000
C       26.45300000    15.64600000    -7.03100000
C       26.60100000    16.44500000    -4.74700000
O       26.50400000    15.29500000    -4.32700000
N       26.55200000    16.73200000    -6.05900000
H       27.09600000    17.17500000    -2.82400000
H       26.55000000    17.70000000    -6.36300000
H       25.59500000    15.01900000    -6.78400000
H       27.40527000    18.39530000    -4.08689700
H       25.72790000    18.02497000    -3.54153100
H       26.29104000    16.06583000    -8.03474900
H       27.33872000    14.99450000    -6.99856900
--
0 1
C       26.29500000    11.22500000    -4.33200000
C       29.47700000    13.05300000    -3.31400000
C       28.49500000    11.91700000    -3.58300000
O       28.77600000    10.75200000    -3.29800000
N       27.31000000    12.25000000    -4.09700000
H       27.09800000    13.22300000    -4.29400000
H       26.10000000    10.69600000    -3.39600000
H       30.46400000    12.61500000    -3.15100000
H       25.36511000    11.71534000    -4.65583800
H       26.63023000    10.48275000    -5.07138000
H       29.52301000    13.76051000    -4.15501800
H       29.15797000    13.56143000    -2.39219900
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '054ARG-062GLN-1')] = qcdb.Molecule("""
0 1
C       -5.93600000    14.71300000    11.54100000
C       -7.26700000    11.13900000    11.56900000
C       -5.94500000    13.19300000    11.45500000
O       -4.90300000    12.57200000    11.30100000
N       -7.11800000    12.58900000    11.57200000
H       -6.95800000    15.07200000    11.52700000
H       -7.94600000    13.16400000    11.70600000
H       -6.33700000    10.65900000    11.27000000
H       -5.45554200    15.04857000    12.47189000
H       -5.42263200    15.16711000    10.68063000
H       -7.55862900    10.74068000    12.55200000
H       -8.02788900    10.89217000    10.81394000
--
0 1
C       -1.35200000    12.46600000    11.67100000
C       -3.58200000     9.31000000    11.71300000
C       -2.31700000    10.18000000    11.90600000
O       -1.24100000     9.68200000    12.23900000
N       -2.45400000    11.49100000    11.70900000
H       -3.37700000    11.84900000    11.50300000
H       -0.42700000    11.95300000    11.40400000
H       -4.39300000     9.97900000    11.43200000
H       -1.55580100    13.25626000    10.93347000
H       -1.22149900    12.98224000    12.63353000
H       -3.45255300     8.55740200    10.92127000
H       -3.85167500     8.81225000    12.65614000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '055ASN-058GLU-2')] = qcdb.Molecule("""
0 1
C        3.19200000    17.07400000    -9.88600000
C        5.08000000    20.00600000    -8.31000000
C        4.31100000    18.03600000    -9.50600000
O        5.45500000    17.83700000    -9.93200000
N        4.03100000    19.04300000    -8.66900000
H        3.23300000    16.30400000    -9.11900000
H        3.09700000    19.15000000    -8.29000000
H        5.40500000    20.48900000    -9.23400000
H        2.17964000    17.50367000    -9.86338600
H        3.40369000    16.59269000   -10.85219000
H        4.70588400    20.79572000    -7.64188500
H        5.97005700    19.52724000    -7.87573800
--
0 1
C        9.03100000    16.38600000    -9.35400000
C        7.29200000    17.70300000    -6.23800000
C        8.40400000    17.23400000    -7.16100000
O        9.55900000    17.14700000    -6.74400000
N        8.06200000    16.93000000    -8.41100000
H        7.09600000    17.02900000    -8.69800000
H        9.54200000    15.53700000    -8.89500000
H        7.70000000    18.51200000    -5.63400000
H        8.51880300    16.03279000   -10.26114000
H        9.80257100    17.12296000    -9.62152600
H        6.43562400    18.10098000    -6.80212100
H        7.02139000    16.89545000    -5.54184000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '060LEU-064TYR-1')] = qcdb.Molecule("""
0 1
C       26.99500000     5.04200000    -5.25200000
C       30.19300000     4.10900000    -7.06200000
C       28.06600000     4.21300000    -5.93800000
O       27.96100000     2.98900000    -6.01300000
N       29.12300000     4.85700000    -6.41300000
H       26.49100000     4.37700000    -4.55500000
H       29.12500000     5.87000000    -6.44800000
H       29.77000000     3.49600000    -7.86000000
H       27.43791000     5.87827900    -4.69122800
H       26.25463000     5.41182500    -5.97662900
H       30.89245000     4.82784200    -7.51370000
H       30.71154000     3.43933300    -6.36010300
--
0 1
C       27.08700000    -0.08000000    -5.13600000
C       29.22300000     1.16000000    -2.22300000
C       28.29800000     0.31100000    -3.06400000
O       27.86700000    -0.74100000    -2.57100000
N       28.02000000     0.70600000    -4.31100000
H       28.33800000     1.60500000    -4.65100000
H       26.87700000    -1.04800000    -4.68000000
H       29.87500000     0.45300000    -1.70700000
H       26.12878000     0.44577620    -5.25996300
H       27.48949000    -0.29528570    -6.13682600
H       29.80564000     1.88508100    -2.81018700
H       28.62863000     1.67276100    -1.45241500
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '060TYR-064LEU-2')] = qcdb.Molecule("""
0 1
C        0.74400000    20.88200000     9.46700000
C        3.82900000    19.04800000     8.16100000
C        1.69600000    19.73500000     9.12300000
O        1.44700000    18.57500000     9.47800000
N        2.82000000    20.06000000     8.47800000
H        0.03500000    20.51800000    10.21200000
H        2.96300000    21.01600000     8.16900000
H        3.36900000    18.28200000     7.53800000
H        1.30436300    21.72894000     9.88971700
H        0.18058210    21.20498000     8.57916800
H        4.65839300    19.49515000     7.59340600
H        4.20918500    18.55045000     9.06537900
--
0 1
C        1.86700000    14.55400000     9.81100000
C        1.62200000    17.04600000    12.67600000
C        1.63700000    15.71000000    11.95000000
O        1.54000000    14.64800000    12.57400000
N        1.78300000    15.76400000    10.62500000
H        1.82800000    16.67400000    10.17600000
H        0.95000000    13.97500000     9.93200000
H        1.73100000    16.86900000    13.74700000
H        0.65600000    17.52100000    12.49900000
H        2.00216700    14.80708000     8.74907700
H        2.69819000    13.91906000    10.15156000
H        2.41953600    17.71840000    12.32698000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '061VAL-064TYR-2')] = qcdb.Molecule("""
0 1
C       27.19000000     8.05900000    14.90900000
C       30.05500000     6.38400000    13.02700000
C       28.60200000     7.63200000    14.54300000
O       29.56400000     7.99700000    15.22000000
N       28.74700000     6.84300000    13.48900000
H       26.49400000     7.64200000    14.18100000
H       27.92900000     6.62000000    12.93000000
H       30.58500000     5.91700000    13.86500000
H       26.96984000     7.69081000    15.92190000
H       27.09454000     9.15434600    14.87576000
H       29.91740000     5.63481800    12.23341000
H       30.66991000     7.21521400    12.65154000
--
0 1
C       31.65200000    11.35800000    15.03100000
C       33.02500000     8.66700000    12.69000000
C       32.80300000     9.95400000    13.45100000
O       33.58700000    10.87600000    13.23500000
N       31.82100000    10.08800000    14.34800000
H       31.21400000     9.31100000    14.57600000
H       31.75600000    12.16100000    14.29800000
H       34.03700000     8.31700000    12.89800000
H       32.94400000     8.86800000    11.62100000
H       30.63884000    11.44243000    15.45098000
H       32.39121000    11.52107000    15.82911000
H       32.30761000     7.89313400    13.00061000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '062LEU-066LYS-2')] = qcdb.Molecule("""
0 1
C       17.65400000    16.07700000    -4.70500000
C       21.17800000    15.82600000    -3.21600000
C       18.76700000    15.83400000    -3.68000000
O       18.49900000    15.43400000    -2.54700000
N       20.02000000    16.06100000    -4.08500000
H       17.53500000    17.15800000    -4.78700000
H       20.17300000    16.38000000    -5.03300000
H       21.16100000    14.77700000    -2.92300000
H       17.92643000    15.67964000    -5.69388200
H       16.70656000    15.64038000    -4.35613000
H       22.11540000    16.07129000    -3.73669600
H       21.14046000    16.40209000    -2.27967200
--
0 1
C       19.20600000    13.81200000     0.82900000
C       17.65800000    17.31000000     0.68300000
C       18.11400000    15.96100000     1.23700000
O       17.91100000    15.67800000     2.41500000
N       18.72500000    15.12600000     0.39600000
H       18.82400000    15.39200000    -0.58000000
H       18.40600000    13.30400000     1.37200000
H       17.51800000    17.96400000     1.54500000
H       19.48015000    13.19193000    -0.03723072
H       20.05264000    13.88387000     1.52759000
H       18.40326000    17.75670000     0.00843105
H       16.69298000    17.18229000     0.17071660
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '064GLN-067ARG-1')] = qcdb.Molecule("""
0 1
C       10.81800000   -25.83400000    -6.60000000
C       14.08600000   -27.13200000    -5.21000000
C       11.83700000   -26.33500000    -5.59400000
O       11.54600000   -26.46100000    -4.39700000
N       13.03400000   -26.64400000    -6.08700000
H        9.98800000   -25.39600000    -6.04100000
H       13.19300000   -26.62000000    -7.08800000
H       13.69600000   -27.96300000    -4.62000000
H       11.25422000   -25.07493000    -7.26597200
H       10.44391000   -26.68721000    -7.18487700
H       14.94122000   -27.49924000    -5.79629000
H       14.40518000   -26.34161000    -4.51472000
--
0 1
C       11.57400000   -23.68400000    -1.98900000
C       14.96000000   -23.69900000    -3.79900000
C       13.93100000   -23.61200000    -2.67200000
O       14.30600000   -23.41800000    -1.51800000
N       12.64300000   -23.71300000    -3.00300000
H       12.39500000   -23.79800000    -3.98800000
H       11.64600000   -22.75300000    -1.41700000
H       15.93300000   -23.88700000    -3.34000000
H       10.58182000   -23.72648000    -2.46204900
H       11.68063000   -24.50391000    -1.26347700
H       14.71306000   -24.52340000    -4.48411700
H       15.01139000   -22.74833000    -4.34998800
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '064TYR-067GLU-1')] = qcdb.Molecule("""
0 1
C       31.49500000   -10.17100000     9.56100000
C       34.83300000   -11.75500000     8.70300000
C       32.51500000   -11.24200000     9.19800000
O       32.14300000   -12.38700000     8.95000000
N       33.79300000   -10.86500000     9.14600000
H       32.01200000    -9.26900000     9.87800000
H       34.02500000    -9.90000000     9.35500000
H       35.81500000   -11.28900000     8.79700000
H       34.80600000   -12.69700000     9.25100000
H       30.89282000    -9.98786000     8.65886700
H       30.83944000   -10.49806000    10.38153000
H       34.59261000   -11.90902000     7.64069500
--
0 1
C       30.20800000   -14.32300000     6.30500000
C       33.86000000   -13.48900000     5.38200000
C       32.37800000   -13.83100000     5.15300000
O       31.98700000   -14.10000000     3.99900000
N       31.59000000   -13.83200000     6.24500000
H       31.99200000   -13.51000000     7.11600000
H       30.03400000   -14.98100000     5.45600000
H       34.05100000   -12.60000000     4.78200000
H       30.04722000   -14.86880000     7.24640700
H       29.53462000   -13.45661000     6.22794000
H       34.11625000   -13.28766000     6.43261900
H       34.45950000   -14.29445000     4.93272600
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '066PHE-072THR-1')] = qcdb.Molecule("""
0 1
C       38.12400000    19.12600000    10.45700000
C       41.27800000    17.83300000    12.11600000
C       39.19300000    18.88700000    11.51900000
O       39.06900000    19.36600000    12.67200000
N       40.24100000    18.16000000    11.15200000
H       37.95200000    18.18900000     9.92300000
H       40.33900000    17.83600000    10.19900000
H       40.81700000    17.65000000    13.08300000
H       37.19343000    19.45829000    10.94034000
H       38.45213000    19.89211000     9.73908200
H       41.82961000    16.91696000    11.85792000
H       41.95229000    18.69250000    12.24478000
--
0 1
C       38.70100000    16.94300000    15.29800000
C       35.73100000    19.33300000    15.56700000
C       37.41600000    17.65600000    14.93500000
O       36.80400000    17.37900000    13.92800000
N       36.99200000    18.60300000    15.75300000
H       39.39200000    17.17500000    14.49400000
H       37.52600000    18.79100000    16.59100000
H       35.02500000    18.70900000    15.02300000
H       35.30000000    19.53400000    16.54900000
H       39.13196000    17.25018000    16.26232000
H       38.52783000    15.85691000    15.27771000
H       35.82698000    20.29205000    15.03690000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '068ASP-072GLN-1')] = qcdb.Molecule("""
0 1
C      -12.64500000    14.92300000   -14.97700000
C      -16.34600000    14.13400000   -14.69400000
C      -14.15200000    15.06000000   -15.06700000
O      -14.65100000    16.10000000   -15.49400000
N      -14.89300000    14.05000000   -14.62400000
H      -12.29300000    15.94900000   -15.06800000
H      -14.45500000    13.20100000   -14.28000000
H      -16.65500000    14.38200000   -15.70700000
H      -12.33835000    14.54139000   -13.99194000
H      -12.25022000    14.30387000   -15.79604000
H      -16.81552000    13.18993000   -14.38051000
H      -16.72844000    14.97696000   -14.09973000
--
0 1
C      -17.32700000    18.82000000   -17.09600000
C      -15.11100000    19.30700000   -14.01600000
C      -15.87100000    19.62100000   -15.30300000
O      -15.80500000    20.74200000   -15.80400000
N      -16.59700000    18.64600000   -15.84000000
H      -16.64500000    17.75100000   -15.37100000
H      -16.67500000    19.35400000   -17.79000000
H      -15.29900000    20.13200000   -13.32900000
H      -17.54602000    17.83440000   -17.53260000
H      -18.24433000    19.41767000   -16.98968000
H      -15.42652000    18.35970000   -13.55440000
H      -14.02744000    19.29478000   -14.20510000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '071GLU-075ASP-1')] = qcdb.Molecule("""
0 1
C       15.57200000    63.34800000    -3.02500000
C       15.98300000    65.88800000    -0.20200000
C       15.59700000    64.69300000    -2.29900000
O       15.32100000    65.73700000    -2.89600000
N       15.95100000    64.67600000    -1.01400000
H       15.75800000    63.53700000    -4.08400000
H       16.16700000    63.79100000    -0.57200000
H       15.00000000    66.35600000    -0.26900000
H       16.33731000    62.66589000    -2.62622700
H       14.56696000    62.91645000    -2.90815100
H       16.15603000    65.64604000     0.85701650
H       16.71665000    66.62310000    -0.56448320
--
0 1
C       13.57000000    68.34300000    -4.88600000
C       17.29200000    67.52600000    -5.37000000
C       15.97900000    68.31700000    -5.39800000
O       15.95400000    69.45600000    -5.87200000
N       14.89100000    67.71000000    -4.92400000
H       15.00600000    66.83100000    -4.43400000
H       13.70300000    69.39800000    -4.65200000
H       18.05100000    68.16800000    -5.81900000
H       13.06464000    68.25306000    -5.85889300
H       12.95575000    67.90806000    -4.08380200
H       17.60412000    67.24020000    -4.35466700
H       17.23709000    66.62674000    -6.00112200
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '072ASN-075ARG-2')] = qcdb.Molecule("""
0 1
C       33.89600000     6.61200000     3.83600000
C       34.69000000     4.61600000     6.97000000
C       34.42900000     5.50500000     4.73100000
O       35.00900000     4.53800000     4.23900000
N       34.23500000     5.64000000     6.03900000
H       33.70400000     6.16000000     2.86000000
H       33.83900000     6.50000000     6.40300000
H       35.72900000     4.36500000     6.75300000
H       32.95879000     7.03835000     4.22311000
H       34.65839000     7.39666800     3.72174900
H       34.60598000     4.97937900     8.00484100
H       34.07659000     3.71393200     6.82855600
--
0 1
C       32.89300000     1.83500000     2.75400000
C       31.64600000     2.43700000     6.29600000
C       32.07900000     1.69100000     5.04400000
O       32.07800000     0.45900000     5.01100000
N       32.45500000     2.44000000     4.00800000
H       32.35000000     3.45000000     4.07300000
H       32.11400000     1.16700000     2.37800000
H       31.68400000     1.74500000     7.13900000
H       33.09690000     2.61038500     2.00087100
H       33.79963000     1.23396200     2.91765600
H       32.32449000     3.28473700     6.47204500
H       30.61180000     2.78413400     6.15484200
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '072THR-075PRO-1')] = qcdb.Molecule("""
0 1
C       24.87800000    30.59800000    37.00200000
C       23.98600000    30.69500000    40.66600000
C       24.70900000    30.14000000    38.42400000
O       25.02000000    28.99400000    38.74700000
N       24.27500000    31.03500000    39.28700000
H       25.20800000    31.63500000    37.01100000
H       24.01800000    31.95400000    38.95000000
H       24.23600000    29.65300000    40.87700000
H       25.65233000    29.97188000    36.53468000
H       23.88452000    30.55410000    36.53182000
H       24.56616000    31.33691000    41.34524000
H       22.90592000    30.80656000    40.84203000
--
0 1
C       20.51900000    27.60800000    41.28700000
C       24.09000000    26.46200000    40.66400000
C       21.87700000    27.50400000    40.59800000
O       22.12000000    28.12800000    39.56400000
N       22.75700000    26.70500000    41.19600000
H       20.67700000    27.73700000    42.35700000
H       22.47800000    26.20800000    42.03200000
H       24.52600000    27.41400000    40.38900000
H       19.88837000    28.40174000    40.86006000
H       19.91999000    26.69611000    41.14682000
H       24.71285000    25.99288000    41.43987000
H       24.03805000    25.80829000    39.78084000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '081ASN-084GLY-1')] = qcdb.Molecule("""
0 1
C       13.82900000     2.70400000    76.43400000
C       13.74200000     6.48100000    76.11400000
C       13.94200000     4.07700000    75.80300000
O       14.21100000     4.22600000    74.61600000
N       13.73500000     5.11200000    76.60900000
H       12.78400000     2.62700000    76.73200000
H       13.50700000     4.96200000    77.58400000
H       13.09800000     6.53000000    75.23300000
H       14.06479000     1.88174600    75.74241000
H       14.39647000     2.61560100    77.37217000
H       13.31125000     7.15543700    76.86871000
H       14.73520000     6.83877900    75.80488000
--
0 1
C       16.99800000     4.23000000    72.56000000
C       17.53300000     6.51800000    75.52300000
C       17.78400000     5.81300000    74.20700000
O       18.82500000     5.98600000    73.58000000
N       16.81400000     5.03000000    73.76300000
H       15.95700000     4.92100000    74.29200000
H       16.01600000     3.95900000    72.17000000
H       17.51700000     4.81600000    71.81200000
H       17.64000000     7.58500000    75.31700000
H       17.56474000     3.30234800    72.72814000
H       16.52165000     6.31557900    75.90535000
H       18.29329000     6.23713200    76.26669000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '081LEU-084LYS-1')] = qcdb.Molecule("""
0 1
C       34.35100000    -7.61400000     4.94900000
C       33.84100000    -8.35400000     1.25300000
C       34.10200000    -8.39000000     3.66300000
O       33.98400000    -9.61500000     3.68700000
N       34.03300000    -7.68800000     2.53300000
H       34.86500000    -8.27200000     5.65200000
H       34.05500000    -6.67200000     2.57800000
H       32.97500000    -9.01300000     1.33800000
H       34.99681000    -6.74889700     4.73799900
H       33.40068000    -7.27977600     5.39080700
H       33.67216000    -7.61640600     0.45459590
H       34.71946000    -8.97837600     1.03284400
--
0 1
C       36.79800000   -11.79300000     4.08200000
C       37.46400000    -9.52200000     1.09100000
C       37.42500000   -10.78000000     1.95200000
O       37.85300000   -11.85300000     1.52300000
N       36.89700000   -10.65500000     3.17100000
H       36.63200000    -9.73000000     3.49300000
H       37.78100000   -12.25100000     4.21000000
H       37.51400000    -9.83900000     0.05000000
H       36.42100000   -11.44903000     5.05645300
H       36.12353000   -12.54555000     3.64753300
H       36.54672000    -8.93834700     1.25818800
H       38.35989000    -8.93906600     1.35095700
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '082LEU-106LEU-2')] = qcdb.Molecule("""
0 1
C       21.00100000     3.07600000     2.09400000
C       23.62700000     5.79100000     2.46200000
C       22.97000000     4.43000000     2.56200000
O       23.55000000     3.50500000     3.13600000
N       21.76000000     4.31500000     2.03200000
H       21.33700000     5.11600000     1.57200000
H       21.65400000     2.23300000     2.32300000
H       22.86400000     6.54500000     2.28500000
H       20.22093000     3.15573100     2.86545000
H       20.51025000     2.86982600     1.13137100
H       24.11967000     5.97366700     3.42839000
H       24.35603000     5.81690800     1.63869000
--
0 1
C       19.26900000     7.51000000    -0.68000000
C       21.81600000     9.19600000     1.55000000
C       20.33800000     7.72700000     0.37400000
O       20.75800000     6.76700000     1.03200000
N       20.79800000     8.95300000     0.54500000
H       19.22500000     8.35300000    -1.37100000
H       20.46300000     9.71700000    -0.03400000
H       21.81500000     8.41000000     2.30200000
H       22.79900000     9.21800000     1.08000000
H       19.63525000     6.59249100    -1.16377800
H       18.30339000     7.33972600    -0.18139080
H       21.66437000    10.13711000     2.09892200
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '084LEU-088MET-2')] = qcdb.Molecule("""
0 1
C       29.89300000    14.63600000     9.04800000
C       31.80200000    14.09700000     5.81700000
C       30.72900000    14.90100000     7.81300000
O       30.99800000    16.05300000     7.49900000
N       31.11900000    13.86900000     7.07800000
H       29.32900000    15.54900000     9.23200000
H       30.92400000    12.92200000     7.38200000
H       32.67500000    14.72400000     6.00500000
H       29.19122000    13.80820000     8.86840500
H       30.52296000    14.42170000     9.92391000
H       32.11987000    13.15770000     5.34090500
H       31.18783000    14.67492000     5.11073800
--
0 1
C       31.99300000    19.25900000     5.79300000
C       28.32400000    18.23400000     6.10200000
C       29.56400000    19.07200000     5.76300000
O       29.46100000    20.20900000     5.31300000
N       30.75400000    18.52500000     6.01700000
H       30.79600000    17.59500000     6.41900000
H       31.92800000    20.19700000     6.33900000
H       27.46500000    18.71500000     5.63300000
H       32.86196000    18.71177000     6.18726400
H       32.14801000    19.50673000     4.73252900
H       28.42025000    17.21390000     5.70184100
H       28.15506000    18.20895000     7.18866100
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '087ALA-171PRO-2')] = qcdb.Molecule("""
0 1
C       24.55400000    -8.66200000    -7.54600000
C       22.97800000    -5.22600000    -8.00400000
C       24.01800000    -6.29500000    -7.69600000
O       25.16100000    -5.97900000    -7.36900000
N       23.63300000    -7.56300000    -7.82200000
H       22.67300000    -7.78000000    -8.08100000
H       24.92500000    -8.56000000    -6.52500000
H       21.98900000    -5.65100000    -7.84900000
H       24.02034000    -9.62042000    -7.62749100
H       25.42861000    -8.66200000    -8.21312500
H       23.12578000    -4.37475800    -7.32315700
H       23.06799000    -4.89555600    -9.04932700
--
0 1
C       19.11500000   -10.06900000    -8.07800000
C       20.12900000    -7.30800000    -5.66900000
C       19.85500000    -8.85000000    -7.54200000
O       20.74200000    -8.30100000    -8.19900000
N       19.48000000    -8.42900000    -6.34300000
H       18.05200000    -9.88400000    -7.98300000
H       18.72600000    -8.91600000    -5.86100000
H       21.11400000    -7.12200000    -6.09600000
H       19.44272000   -10.35012000    -9.08971800
H       19.32640000   -10.97752000    -7.49499500
H       19.53978000    -6.38066000    -5.72250000
H       20.26252000    -7.57213400    -4.60956300
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '087LEU-090TYR-1')] = qcdb.Molecule("""
0 1
C       16.60300000     4.96400000    15.18700000
C       17.39400000     1.45500000    13.87700000
C       17.06600000     3.84500000    14.25000000
O       17.51400000     4.11600000    13.13000000
N       16.95600000     2.59300000    14.69100000
H       16.35700000     5.82500000    14.56300000
H       16.62200000     2.42300000    15.63700000
H       18.46200000     1.55600000    13.67500000
H       15.70522000     4.71654300    15.77245000
H       17.41544000     5.25344000    15.86976000
H       17.23827000     0.51629330    14.42888000
H       16.89844000     1.39674200    12.89668000
--
0 1
C       17.12600000     4.55000000     9.52800000
C       14.78000000     2.00000000    11.15000000
C       15.51000000     2.78300000    10.05500000
O       15.30400000     2.54800000     8.86000000
N       16.37800000     3.70700000    10.45900000
H       16.52900000     3.83900000    11.45300000
H       16.58700000     4.62100000     8.58400000
H       14.65700000     0.97600000    10.79500000
H       17.19936000     5.56695400     9.94082300
H       18.13360000     4.17566500     9.29431100
H       15.34760000     1.99321600    12.09223000
H       13.78416000     2.43425000    11.32241000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '088PHE-091ALA-1')] = qcdb.Molecule("""
0 1
C        8.88700000    97.91800000    24.54500000
C        7.01100000   100.71100000    26.37800000
C        7.70500000    98.73200000    25.14100000
O        6.52300000    98.34800000    25.07500000
N        8.03300000    99.87300000    25.76400000
H        8.51900000    97.09500000    23.92500000
H        9.00800000   100.11800000    25.90600000
H        6.44000000   100.08900000    27.06600000
H        9.54203400    98.58050000    23.96017000
H        9.47528500    97.49463000    25.37245000
H        7.46875000   101.54240000    26.93414000
H        6.26045700   101.06570000    25.65630000
--
0 1
C        4.57800000    98.59500000    21.82800000
C        5.46800000   102.10100000    23.16500000
C        4.66900000   100.93200000    22.55600000
O        3.47300000   101.09900000    22.25300000
N        5.28300000    99.74400000    22.40700000
H        6.25700000    99.63200000    22.66800000
H        4.14500000    98.89200000    20.87200000
H        4.71700000   102.75200000    23.61500000
H        5.24695200    97.73820000    21.65951000
H        3.75337300    98.29375000    22.49075000
H        6.17450700   101.76120000    23.93662000
H        5.99833600   102.66930000    22.38670000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '089MET-093GLY-1')] = qcdb.Molecule("""
0 1
C       10.15500000    28.16600000    14.94000000
C        8.42100000    30.63000000    12.60000000
C        9.71300000    29.47500000    14.29900000
O       10.00700000    30.54700000    14.83800000
N        8.95800000    29.40800000    13.20400000
H       10.34700000    28.38800000    15.99200000
H        8.77300000    28.50600000    12.77000000
H        9.24600000    31.30200000    12.35200000
H        9.36884600    27.39869000    14.88347000
H       11.09582000    27.80197000    14.50143000
H        7.86333900    30.40004000    11.68015000
H        7.77766200    31.14956000    13.32537000
--
0 1
C        9.76000000    34.38600000    15.64000000
C        8.28300000    31.69800000    17.90400000
C        8.92900000    33.03900000    17.51800000
O        9.03200000    33.95700000    18.33700000
N        9.40100000    33.13300000    16.28000000
H        9.34800000    32.30800000    15.69300000
H        9.66400000    34.25100000    14.56200000
H        9.03300000    35.14400000    15.93400000
H        7.65100000    31.85800000    18.77800000
H       10.76051000    34.80161000    15.83037000
H        7.66408600    31.29681000    17.08792000
H        9.06646600    30.96557000    18.14838000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '092SER-096ARG-2')] = qcdb.Molecule("""
0 1
C       20.72500000     2.77400000    15.80200000
C       22.42300000     0.67100000    13.12000000
C       21.05600000     2.18900000    14.44300000
O       20.40600000     2.51900000    13.45700000
N       22.05700000     1.30800000    14.38000000
H       19.64600000     2.88300000    15.86900000
H       22.58100000     1.08300000    15.21400000
H       22.56400000     1.44400000    12.36100000
H       21.06547000     2.09683400    16.59920000
H       21.17889000     3.77111700    15.90070000
H       23.36780000     0.11837040    13.22937000
H       21.61754000     0.00530976    12.77635000
--
0 1
C       18.67300000     2.67700000    10.22100000
C       16.97100000     1.13700000    13.25700000
C       17.14200000     1.63400000    11.82800000
O       16.20400000     1.61900000    11.03600000
N       18.35800000     2.05700000    11.50300000
H       19.07400000     2.05600000    12.22600000
H       17.84100000     3.31800000     9.91700000
H       15.98600000     0.67700000    13.32300000
H       19.57535000     3.29629000    10.33164000
H       18.81920000     1.97051000     9.39064000
H       17.73756000     0.39736570    13.53146000
H       17.00310000     2.00729400    13.92898000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '095GLN-183ILE-1')] = qcdb.Molecule("""
0 1
C       12.33100000     5.30600000   -10.12800000
C       15.66800000     3.88400000   -11.25800000
C       14.34800000     4.62700000   -11.27400000
O       13.99100000     5.23600000   -12.30100000
N       13.62400000     4.61400000   -10.15500000
H       13.95200000     4.12300000    -9.32700000
H       12.11500000     5.75900000   -11.09500000
H       15.74200000     3.31400000   -10.33200000
H       11.49792000     4.62935000    -9.88690300
H       12.32161000     6.12967300    -9.39897700
H       15.69438000     3.20422600   -12.12241000
H       16.51174000     4.58604700   -11.33028000
--
0 1
C       15.18400000     3.83200000    -5.59000000
C       16.79300000     1.28500000    -7.88000000
C       15.47000000     3.00800000    -6.82200000
O       14.80600000     3.17100000    -7.84900000
N       16.48200000     2.14700000    -6.75500000
H       16.06900000     3.81600000    -4.95000000
H       17.01500000     2.05300000    -5.89900000
H       16.73600000     1.86300000    -8.80400000
H       14.94688000     4.85878500    -5.90541300
H       14.34217000     3.38624000    -5.03989800
H       17.81981000     0.90047880    -7.79170300
H       16.03889000     0.48457490    -7.90533000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '095LYS-107ILE-2')] = qcdb.Molecule("""
0 1
C       16.07000000    16.05100000    25.83200000
C       18.67100000    13.26400000    25.80500000
C       17.34300000    13.97600000    26.07900000
O       16.40700000    13.39700000    26.63500000
N       17.26500000    15.23200000    25.65600000
H       18.11300000    15.66400000    25.29800000
H       15.29700000    15.51300000    26.38300000
H       19.39800000    14.01400000    25.49000000
H       16.33112000    16.97462000    26.36934000
H       15.61791000    16.31006000    24.86324000
H       19.05395000    12.74713000    26.69730000
H       18.55279000    12.54318000    24.98254000
--
0 1
C       20.35100000    18.72600000    24.18900000
C       21.45900000    16.82300000    27.28600000
C       20.41400000    17.64700000    25.24500000
O       19.66200000    16.67700000    25.17100000
N       21.30700000    17.80200000    26.21600000
H       21.18100000    19.40800000    24.34400000
H       21.86700000    18.64900000    26.23700000
H       20.60400000    16.16500000    27.27600000
H       20.41973000    18.27131000    23.18973000
H       19.43017000    19.31796000    24.29695000
H       22.38030000    16.23033000    27.18626000
H       21.53046000    17.30278000    28.27327000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '097GLU-100THR-2')] = qcdb.Molecule("""
0 1
C        9.39000000    -7.30300000    29.75300000
C       11.89300000    -4.60400000    28.68400000
C        9.94400000    -6.02200000    29.10700000
O        9.23700000    -5.38500000    28.30400000
N       11.21300000    -5.68900000    29.38900000
H        8.81200000    -7.81300000    28.97900000
H       11.73900000    -6.21700000    30.07300000
H       11.35200000    -3.67100000    28.83400000
H       10.16424000    -8.00472900    30.09669000
H        8.70615500    -7.07017100    30.58255000
H       12.92471000    -4.48928400    29.04788000
H       11.93164000    -4.81763100    27.60564000
--
0 1
C        8.28800000    -6.21100000    24.73100000
C       12.02100000    -6.57700000    25.44600000
C       10.70500000    -6.50700000    24.70400000
O       10.70400000    -6.65400000    23.48200000
N        9.60400000    -6.26900000    25.39600000
H        9.64500000    -6.19000000    26.40600000
H        8.39900000    -6.31300000    23.66100000
H       12.67400000    -5.86700000    24.93100000
H        7.63027500    -7.02978600    25.05809000
H        7.80957600    -5.23635300    24.90756000
H       11.97940000    -6.28878400    26.50675000
H       12.45965000    -7.57595600    25.30575000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '102GLN-106ILE-1')] = qcdb.Molecule("""
0 1
C       15.81800000   -10.94900000     7.38500000
C       15.16100000   -12.86700000    10.61500000
C       15.07000000   -11.52800000     8.59900000
O       13.88700000   -11.25200000     8.80700000
N       15.74900000   -12.35700000     9.38900000
H       15.34400000   -10.00200000     7.11800000
H       16.72300000   -12.54500000     9.20000000
H       14.20200000   -13.33800000    10.39200000
H       16.88595000   -10.75380000     7.56211300
H       15.74583000   -11.63969000     6.53192300
H       15.81762000   -13.61630000    11.08126000
H       14.98669000   -12.01714000    11.29127000
--
0 1
C       10.49500000   -10.10800000    10.72100000
C       13.37800000    -7.68100000    10.07700000
C       11.98600000    -8.21800000    10.41200000
O       11.07200000    -7.44000000    10.67500000
N       11.80700000    -9.53900000    10.39700000
H       12.56900000   -10.15000000    10.12400000
H        9.73000000    -9.44900000    10.32500000
H       13.45100000    -6.68400000    10.51600000
H       10.31506000   -11.09194000    10.26330000
H       10.31658000   -10.16052000    11.80516000
H       14.18648000    -8.29659100    10.49819000
H       13.50531000    -7.57943600     8.98912300
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '104VAL-108ILE-1')] = qcdb.Molecule("""
0 1
C       37.23300000    -8.66000000    10.91900000
C       38.92400000   -12.07000000    10.79300000
C       38.24200000    -9.74400000    10.55300000
O       39.19900000    -9.49700000     9.81500000
N       38.01900000   -10.95400000    11.06000000
H       37.41800000    -7.80600000    10.26900000
H       37.23800000   -11.09100000    11.69500000
H       39.94200000   -11.78700000    11.07200000
H       36.21551000    -9.04550400    10.75743000
H       37.39077000    -8.35867900    11.96510000
H       38.62482000   -12.95752000    11.36989000
H       38.93919000   -12.32176000     9.72230500
--
0 1
C       42.35200000    -9.76400000     7.69200000
C       38.97900000    -9.48700000     5.99100000
C       40.48600000    -9.52100000     6.18000000
O       41.24800000    -9.41500000     5.22100000
N       40.92600000    -9.67400000     7.42400000
H       40.26700000    -9.74200000     8.19500000
H       42.85600000    -8.86000000     7.34500000
H       38.74500000    -9.62600000     4.94300000
H       42.52499000    -9.89266800     8.77066600
H       42.75928000   -10.61663000     7.12884700
H       38.48049000   -10.27524000     6.57423600
H       38.57739000    -8.49842400     6.25826000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '108LYS-112TYR-1')] = qcdb.Molecule("""
0 1
C       -0.54400000     1.81600000    15.86000000
C        1.42400000     4.71400000    14.31800000
C       -0.26800000     3.09400000    15.09800000
O       -1.17400000     3.63000000    14.49200000
N        0.98300000     3.56000000    15.11100000
H       -1.49400000     1.41800000    15.49300000
H        1.66300000     3.07500000    15.68900000
H        0.84600000     5.59700000    14.59600000
H        0.25127980     1.07268500    15.70185000
H       -0.65763980     2.09831300    16.91706000
H        2.48742400     4.91111600    14.51869000
H        1.25912900     4.53704800    13.24492000
--
0 1
C       -2.78600000     6.27300000    12.11600000
C       -2.35400000     2.69900000    10.91500000
C       -2.83200000     4.14500000    10.91300000
O       -3.57300000     4.57400000    10.03300000
N       -2.48700000     4.85700000    11.97400000
H       -1.93800000     4.40900000    12.70200000
H       -3.86500000     6.41500000    12.17000000
H       -2.52100000     2.29300000     9.91500000
H       -2.32373200     6.62150100    13.05134000
H       -2.41530500     6.85011400    11.25604000
H       -1.28375700     2.61886300    11.15616000
H       -2.96919000     2.12759800    11.62566000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '108TYR-129TRP-1')] = qcdb.Molecule("""
0 1
C       21.45900000    16.82300000    27.28600000
C       21.42500000    17.10300000    31.02900000
C       21.55700000    17.48100000    28.64000000
O       21.94300000    18.63900000    28.77400000
N       21.23200000    16.69900000    29.65300000
H       20.60400000    16.16500000    27.27600000
H       20.85600000    15.77500000    29.47000000
H       22.00800000    18.02200000    31.07100000
H       21.34434000    17.56150000    26.47886000
H       22.38030000    16.23033000    27.18626000
H       20.48909000    17.27499000    31.58080000
H       21.98440000    16.31516000    31.55472000
--
0 1
C       24.53000000    20.73600000    27.37500000
C       23.10500000    21.74000000    30.73700000
C       24.01800000    21.66700000    29.53600000
O       25.04200000    22.35200000    29.51200000
N       23.66300000    20.86900000    28.53500000
H       22.83600000    20.28600000    28.59800000
H       25.31900000    21.48700000    27.39300000
H       22.39100000    20.91900000    30.65600000
H       25.01147000    19.74856000    27.31878000
H       23.98691000    20.83089000    26.42313000
H       23.70773000    21.66240000    31.65390000
H       22.56577000    22.69806000    30.70023000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '112SER-115ALA-2')] = qcdb.Molecule("""
0 1
C       19.26500000   -11.56600000    27.68600000
C       17.01200000    -9.19000000    29.53700000
C       18.85500000   -10.36300000    28.48800000
O       19.68000000    -9.66900000    29.06000000
N       17.55000000   -10.18600000    28.61100000
H       18.62000000   -11.61400000    26.80700000
H       16.91200000   -10.76800000    28.07800000
H       17.33900000    -8.18900000    29.24300000
H       20.30707000   -11.47286000    27.34628000
H       19.08773000   -12.47159000    28.28474000
H       15.91356000    -9.24237600    29.51081000
H       17.37671000    -9.36836500    30.55934000
--
0 1
C       21.68800000   -10.28800000    32.04500000
C       18.01100000   -11.01800000    32.71300000
C       19.43000000   -10.50600000    32.92100000
O       19.73200000    -9.87600000    33.94800000
N       20.31500000   -10.75400000    31.95100000
H       20.03400000   -11.27900000    31.12800000
H       22.10200000   -10.61400000    32.99900000
H       17.36000000   -10.52800000    33.43800000
H       22.30271000   -10.72446000    31.24397000
H       21.74199000    -9.18956700    32.02197000
H       17.67236000   -10.77590000    31.69481000
H       17.95706000   -12.10543000    32.86979000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '112TRP-115ARG-1')] = qcdb.Molecule("""
0 1
C       12.99700000   107.58400000    30.66100000
C       11.31500000   109.73200000    33.35200000
C       12.09900000   108.66300000    31.29500000
O       11.42800000   109.42400000    30.60000000
N       12.10600000   108.74900000    32.62400000
H       13.16400000   107.85700000    29.61600000
H       12.64500000   108.07400000    33.15800000
H       10.26900000   109.63600000    33.05700000
H       13.95973000   107.56090000    31.19262000
H       12.50602000   106.59990000    30.68299000
H       11.39995000   109.52640000    34.42927000
H       11.62295000   110.76470000    33.13142000
--
0 1
C       12.56800000   112.51500000    29.00500000
C       13.50400000   112.80200000    32.70700000
C       13.05700000   113.20500000    31.29500000
O       12.82300000   114.38400000    31.01500000
N       12.99300000   112.23500000    30.37600000
H       13.27900000   111.29200000    30.61500000
H       12.88800000   113.52300000    28.73400000
H       13.17300000   113.46900000    33.50200000
H       13.00543000   111.79710000    28.29560000
H       11.47648000   112.52440000    28.86901000
H       13.16244000   111.78340000    32.94300000
H       14.59802000   112.89490000    32.64005000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '113TRP-124HIE-2')] = qcdb.Molecule("""
0 1
C       15.69200000    18.90600000    -3.72600000
C       16.61100000    15.27700000    -4.31600000
C       15.75100000    16.46500000    -3.89100000
O       14.57900000    16.29400000    -3.54500000
N       16.36500000    17.63700000    -3.91800000
H       17.33700000    17.65900000    -4.20600000
H       14.70500000    18.77000000    -3.28900000
H       17.65500000    15.58800000    -4.31600000
H       16.28129000    19.57541000    -3.08208300
H       15.59310000    19.41205000    -4.69766700
H       16.48905000    14.39645000    -3.66811600
H       16.33723000    15.03294000    -5.35305800
--
0 1
C       20.52300000    19.52100000    -4.04000000
C       21.15600000    15.79500000    -3.64800000
C       20.27400000    18.02900000    -4.10900000
O       19.16700000    17.59400000    -4.43900000
N       21.30000000    17.24200000    -3.78000000
H       21.54400000    19.73700000    -4.35300000
H       22.15300000    17.66300000    -3.43100000
H       20.10700000    15.57500000    -3.49900000
H       19.79850000    20.04400000    -4.68153200
H       20.37331000    19.85075000    -3.00132000
H       21.47959000    15.24341000    -4.54300700
H       21.70552000    15.47354000    -2.75095500
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '115GLN-118ARG-2')] = qcdb.Molecule("""
0 1
C       30.32800000    52.65700000    61.95600000
C       26.93700000    54.24800000    61.33000000
C       29.13200000    53.22000000    61.20800000
O       29.09000000    53.17900000    59.97700000
N       28.15300000    53.72900000    61.94300000
H       30.06900000    52.54100000    63.01000000
H       28.25300000    53.77900000    62.95000000
H       26.36300000    53.39700000    60.95800000
H       30.52267000    51.68065000    61.48820000
H       31.19440000    53.32616000    61.84842000
H       26.32112000    54.75007000    62.09067000
H       27.12494000    54.93424000    60.49110000
--
0 1
C       29.73400000    54.20900000    56.40600000
C       26.60100000    55.80500000    57.88100000
C       27.69900000    55.43600000    56.90000000
O       27.69000000    55.91800000    55.76600000
N       28.63800000    54.58200000    57.29900000
H       28.59600000    54.18400000    58.23300000
H       30.02600000    55.10000000    55.83800000
H       25.65200000    55.75600000    57.34500000
H       26.75300000    56.84300000    58.18000000
H       30.61510000    53.88913000    56.98163000
H       29.47935000    53.42987000    55.67244000
H       26.51774000    55.17149000    58.77639000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '119MET-122VAL-1')] = qcdb.Molecule("""
0 1
C       45.95400000    -1.61800000     9.16300000
C       45.36600000     0.73800000     6.24400000
C       45.23200000    -0.89200000     8.03600000
O       44.03200000    -1.09900000     7.82500000
N       45.95700000    -0.02600000     7.33100000
H       45.29500000    -2.41400000     9.51700000
H       46.93200000     0.11700000     7.56300000
H       44.64800000     1.45400000     6.65000000
H       46.15100000     1.29100000     5.72700000
H       46.91743000    -2.05935100     8.86801200
H       46.12011000    -0.91022000     9.98850300
H       44.85330000     0.09929351     5.50970500
--
0 1
C       42.46700000    -3.99300000     6.27200000
C       44.76000000    -2.15600000     3.84700000
C       43.55000000    -2.89700000     4.40500000
O       42.57800000    -3.13000000     3.68800000
N       43.59800000    -3.27700000     5.67900000
H       44.44200000    -3.10200000     6.21900000
H       42.17700000    -4.81600000     5.61600000
H       44.42000000    -1.60800000     2.96500000
H       42.71491000    -4.41473300     7.25723200
H       41.59705000    -3.32680800     6.36882200
H       45.18522000    -1.45155800     4.57703100
H       45.53226000    -2.87443100     3.53479500
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '121LYS-125ALA-1')] = qcdb.Molecule("""
0 1
C       44.76000000    -2.15600000     3.84700000
C       42.46700000    -3.99300000     6.27200000
C       43.55000000    -2.89700000     4.40500000
O       42.57800000    -3.13000000     3.68800000
N       43.59800000    -3.27700000     5.67900000
H       44.42000000    -1.60800000     2.96500000
H       44.44200000    -3.10200000     6.21900000
H       42.17700000    -4.81600000     5.61600000
H       45.18522000    -1.45155800     4.57703100
H       45.53226000    -2.87443100     3.53479500
H       42.71491000    -4.41473300     7.25723200
H       41.59705000    -3.32680800     6.36882200
--
0 1
C       38.70400000    -3.61700000     2.64900000
C       39.95800000    -0.09600000     3.37500000
C       39.08000000    -1.21600000     2.83200000
O       38.06400000    -0.96800000     2.18100000
N       39.46300000    -2.45900000     3.11500000
H       40.33800000    -2.61300000     3.60100000
H       38.33600000    -3.42200000     1.63900000
H       39.36800000     0.82000000     3.44100000
H       40.77600000     0.06800000     2.67200000
H       39.36231000    -4.49666000     2.59581800
H       37.83042000    -3.84151200     3.27864100
H       40.38072000    -0.34283620     4.36008000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '121VAL-155THR-2')] = qcdb.Molecule("""
0 1
C       13.19700000    20.16700000    11.57000000
C       12.05000000    18.43000000     8.37400000
C       12.94300000    18.78300000     9.55300000
O       14.05300000    18.25200000     9.66900000
N       12.45100000    19.68400000    10.40300000
H       11.54000000    20.08900000    10.21300000
H       14.18900000    19.71600000    11.58000000
H       11.24900000    19.16500000     8.31400000
H       13.34590000    21.25582000    11.52204000
H       12.74725000    19.91914000    12.54278000
H       12.64959000    18.47294000     7.45277600
H       11.58365000    17.44310000     8.51019800
--
0 1
C        7.45000000    21.05700000    10.59900000
C       10.02700000    21.93000000     7.94900000
C        8.76700000    21.13800000     9.84500000
O        9.77800000    20.57900000    10.29100000
N        8.78200000    21.82200000     8.70900000
H        6.64500000    21.36600000     9.93700000
H        7.94500000    22.27900000     8.36300000
H       10.75500000    21.22300000     8.33200000
H        7.29745200    20.01786000    10.92600000
H        7.47500000    21.74773000    11.45473000
H        9.88999300    21.63518000     6.89813700
H       10.50478000    22.91227000     8.07891000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '126VAL-129ALA-1')] = qcdb.Molecule("""
0 1
C       19.00600000    31.54700000    21.22500000
C       18.52800000    32.86900000    24.78400000
C       19.26400000    31.80100000    22.71300000
O       20.15100000    31.19300000    23.30700000
N       18.47800000    32.67900000    23.33400000
H       19.39900000    30.55300000    21.01900000
H       17.84800000    33.25700000    22.78700000
H       19.55400000    33.11700000    25.06000000
H       17.92475000    31.55751000    21.02302000
H       19.55905000    32.25316000    20.58824000
H       17.87341000    33.70335000    25.07620000
H       18.25515000    31.94650000    25.31743000
--
0 1
C       19.81200000    27.80600000    24.20100000
C       16.75000000    29.60300000    25.63200000
C       17.94900000    28.64700000    25.54100000
O       18.21800000    27.87900000    26.47100000
N       18.64100000    28.66500000    24.39800000
H       18.30500000    29.22600000    23.62300000
H       19.54200000    26.77300000    24.42800000
H       16.50600000    29.72800000    26.68900000
H       20.14170000    27.85783000    23.15285000
H       20.62281000    28.09665000    24.88519000
H       17.00434000    30.59025000    25.21889000
H       15.87302000    29.16771000    25.13057000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '133LYS-137ASN-1')] = qcdb.Molecule("""
0 1
C        0.71300000     5.33600000    27.76300000
C        1.38700000     1.63800000    28.30600000
C        1.13800000     4.03500000    28.42900000
O        1.68100000     4.05300000    29.54200000
N        0.88400000     2.90600000    27.78200000
H        1.36800000     6.11000000    28.16200000
H        0.36300000     2.94300000    26.91100000
H        1.01500000     1.50900000    29.32400000
H        0.89068970     5.33301400    26.67745000
H       -0.31956720     5.59768800    28.03745000
H        1.03141800     0.78489000    27.70954000
H        2.48548500     1.63657500    28.36370000
--
0 1
C        3.67700000     2.99200000    32.71200000
C        4.91700000     5.31200000    29.93200000
C        4.85700000     4.57500000    31.26000000
O        5.70900000     4.82000000    32.13100000
N        3.84800000     3.70000000    31.42100000
H        3.15200000     3.60500000    30.69000000
H        3.69600000     3.71900000    33.52600000
H        5.90400000     5.76700000    29.84700000
H        2.72434100     2.44288800    32.74227000
H        4.51397500     2.29830400    32.88010000
H        4.76365400     4.63428600    29.07925000
H        4.15324200     6.10359500    29.93914000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '134GLU-138ARG-1')] = qcdb.Molecule("""
0 1
C       -5.47100000    -3.30200000    72.20800000
C       -4.40800000    -0.33600000    74.43800000
C       -5.04100000    -1.87800000    72.63200000
O       -4.79900000    -1.01800000    71.77900000
N       -4.88300000    -1.63800000    73.94100000
H       -5.07000000    -3.43500000    71.20200000
H       -5.10300000    -2.37300000    74.61000000
H       -4.99500000     0.45700000    73.96800000
H       -4.99341800    -4.03843000    72.87101000
H       -6.55733600    -3.46127200    72.14086000
H       -4.53425400    -0.26426490    75.52837000
H       -3.35791000    -0.17223760    74.15429000
--
0 1
C       -3.32900000     2.19200000    70.15700000
C       -1.36500000    -1.07900000    70.06500000
C       -1.77100000     0.34700000    69.75100000
O       -1.13700000     1.00200000    68.92000000
N       -2.85400000     0.81700000    70.37700000
H       -3.37000000     0.20500000    71.00600000
H       -3.57800000     2.32900000    69.10300000
H       -0.33300000    -1.20700000    69.73000000
H       -4.22387700     2.37881900    70.76880000
H       -2.53485900     2.91558200    70.39315000
H       -1.43198700    -1.27996200    71.14441000
H       -2.02256900    -1.75743300    69.50168000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '135ARG-152ALA-2')] = qcdb.Molecule("""
0 1
C       14.42800000     5.43400000    10.89000000
C       14.29100000     4.73900000     7.15400000
C       13.98300000     5.53900000     9.44500000
O       13.08900000     6.31900000     9.12500000
N       14.60500000     4.75100000     8.57800000
H       15.37200000     4.89300000    10.92900000
H       15.32500000     4.12200000     8.92100000
H       13.58900000     5.53700000     6.91700000
H       14.55926000     6.44487500    11.30340000
H       13.67904000     4.88107900    11.47595000
H       15.19575000     4.88384600     6.54536100
H       13.82916000     3.78414200     6.86254000
--
0 1
C        9.61000000     5.47000000     9.33500000
C       10.59400000     8.22700000     6.89600000
C        9.73200000     7.33500000     7.77800000
O        8.50400000     7.47400000     7.78600000
N       10.34800000     6.40400000     8.49400000
H       11.36100000     6.37100000     8.49300000
H        8.54400000     5.56600000     9.15300000
H       11.65700000     8.05600000     7.07800000
H        9.91149400     4.44145900     9.08760200
H        9.74431000     5.66352200    10.40948000
H       10.28615000     7.90699200     5.88960800
H       10.33999000     9.28405700     7.06366100
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '137SER-144LYS-1')] = qcdb.Molecule("""
0 1
C       26.86200000    49.40800000    54.49800000
C       26.15800000    50.39700000    58.07800000
C       26.32700000    49.33100000    55.90900000
O       25.67300000    48.35500000    56.28300000
N       26.61500000    50.35300000    56.70000000
H       27.59300000    50.21400000    54.43300000
H       27.16500000    51.12800000    56.33800000
H       25.72700000    49.44200000    58.37900000
H       27.33000000    48.44171000    54.25870000
H       26.04270000    49.61534000    53.79390000
H       27.01124000    50.60527000    58.74027000
H       25.38427000    51.17435000    58.16204000
--
0 1
C       24.84400000    44.87100000    56.13900000
C       25.45200000    46.36600000    59.60100000
C       25.08800000    45.34100000    58.53500000
O       24.73600000    44.20500000    58.85600000
N       25.16400000    45.74300000    57.27000000
H       25.45500000    46.69700000    57.06600000
H       24.41200000    43.93000000    56.48500000
H       25.70900000    47.30500000    59.11300000
H       24.13207000    45.35928000    55.45728000
H       25.75743000    44.63939000    55.57155000
H       24.59013000    46.50324000    60.27059000
H       26.33313000    46.02721000    60.16565000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '140SER-144THR-2')] = qcdb.Molecule("""
0 1
C      -27.66700000    11.00300000   -22.84300000
C      -25.41500000    13.54300000   -24.57200000
C      -26.71400000    11.61700000   -23.86800000
O      -26.35100000    10.97300000   -24.85200000
N      -26.29700000    12.86100000   -23.63700000
H      -27.71400000     9.92900000   -23.03700000
H      -26.62500000    13.36600000   -22.81800000
H      -25.87100000    13.53300000   -25.56000000
H      -27.30905000    11.15781000   -21.81446000
H      -28.68080000    11.41427000   -22.95732000
H      -25.26131000    14.58709000   -24.26173000
H      -24.45083000    13.01972000   -24.65295000
--
0 1
C      -24.65800000    10.06900000   -28.09100000
C      -23.49800000     8.40600000   -24.86900000
C      -23.63000000     8.68500000   -26.36100000
O      -23.00500000     8.02200000   -27.18500000
N      -24.44400000     9.68100000   -26.70100000
H      -24.99800000    10.13100000   -25.97900000
H      -25.05200000     9.21300000   -28.64100000
H      -22.60600000     7.78500000   -24.74500000
H      -25.39211000    10.88515000   -28.16163000
H      -23.71053000    10.36369000   -28.56582000
H      -23.37525000     9.34731400   -24.31325000
H      -24.37686000     7.85338700   -24.50538000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '142ALA-146PHE-1')] = qcdb.Molecule("""
0 1
C       33.55000000    94.86800000    46.73100000
C       35.97100000    97.65100000    45.54600000
C       34.40300000    95.74100000    45.79100000
O       34.35300000    95.53500000    44.57400000
N       35.18700000    96.67900000    46.34300000
H       33.80500000    93.83000000    46.52600000
H       35.16900000    96.78300000    47.35400000
H       35.30500000    98.17100000    44.85500000
H       33.70052000    94.99519000    47.81321000
H       32.48246000    95.00923000    46.50648000
H       36.43341000    98.39401000    46.21242000
H       36.72009000    97.21313000    44.86988000
--
0 1
C       34.76000000    95.95100000    40.63600000
C       36.62700000    93.16800000    42.46700000
C       36.01800000    94.02700000    41.34400000
O       36.18100000    93.72900000    40.15200000
N       35.32400000    95.12000000    41.69400000
H       35.06000000    95.30900000    42.65700000
H       34.32600000    95.29100000    39.88900000
H       37.39000000    92.50800000    42.05100000
H       33.97288000    96.62404000    41.00675000
H       35.52326000    96.46537000    40.03362000
H       37.12414000    93.80579000    43.21270000
H       35.81858000    92.54917000    42.88354000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '143VAL-147GLU-1')] = qcdb.Molecule("""
0 1
C       -1.86200000     9.06900000    -7.73200000
C       -1.72900000     5.24100000    -7.46700000
C       -1.43000000     7.67500000    -7.27000000
O       -0.51700000     7.55000000    -6.45200000
N       -2.05900000     6.63000000    -7.81800000
H       -1.07700000     9.76500000    -7.43100000
H       -2.84200000     6.81500000    -8.44200000
H       -1.81000000     5.11300000    -6.38400000
H       -1.95655300     9.09526500    -8.82761400
H       -2.79855100     9.37478300    -7.24274600
H       -2.42071900     4.53998200    -7.95699800
H       -0.69671890     4.97302500    -7.73641600
--
0 1
C        1.54100000     5.71700000    -4.03200000
C        3.10700000     7.59300000    -6.97000000
C        3.01800000     6.83000000    -5.64900000
O        4.03100000     6.57600000    -5.00000000
N        1.80200000     6.44000000    -5.27500000
H        1.00400000     6.73000000    -5.83200000
H        1.80100000     6.35800000    -3.19200000
H        4.15000000     7.55900000    -7.29000000
H        0.46699580     5.48783100    -3.96878100
H        2.13326100     4.79419200    -3.94452300
H        2.48037000     7.11195900    -7.73546400
H        2.83065700     8.64838900    -6.82932900
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '146PHE-150LEU-1')] = qcdb.Molecule("""
0 1
C       35.29100000     3.15000000    19.07200000
C       32.79300000     2.00300000    21.70800000
C       34.58400000     3.01000000    20.42000000
O       35.10400000     3.49300000    21.41900000
N       33.44500000     2.30600000    20.44100000
H       36.34200000     3.36700000    19.26800000
H       33.03500000     1.97800000    19.57100000
H       32.55100000     2.94400000    22.20500000
H       35.22787000     2.22328600    18.48275000
H       34.86745000     3.99197100    18.50482000
H       31.85335000     1.46779100    21.50649000
H       33.49182000     1.45675800    22.35860000
--
0 1
C       36.08300000     4.39300000    24.97900000
C       38.17900000     2.13700000    22.73200000
C       37.81100000     3.06600000    23.90200000
O       38.61900000     3.32300000    24.79600000
N       36.58400000     3.58100000    23.89100000
H       35.97900000     3.40000000    23.09600000
H       36.81400000     5.16900000    25.19500000
H       39.05100000     1.55200000    23.02500000
H       35.12747000     4.86644900    24.70918000
H       35.98399000     3.81770400    25.91133000
H       37.36631000     1.44696500    22.46110000
H       38.44675000     2.74302700    21.85391000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '150LYS-158LEU-2')] = qcdb.Molecule("""
0 1
C        9.85300000     2.84200000    -3.55200000
C       10.78000000     4.84700000    -0.46600000
C       10.33700000     4.01700000    -2.71400000
O       10.59100000     5.10800000    -3.23200000
N       10.47800000     3.78100000    -1.41700000
H        9.31000000     2.16500000    -2.88600000
H       10.26800000     2.86500000    -1.04700000
H       11.01900000     5.76500000    -0.99800000
H        9.16822400     3.17310000    -4.34664100
H       10.71315000     2.29424100    -3.96443900
H        9.91530300     5.05294900     0.18198460
H       11.64222000     4.62279400     0.17922190
--
0 1
C       13.85500000     7.24900000    -3.77000000
C       10.43300000     8.49100000    -2.65800000
C       11.93400000     8.51000000    -2.95700000
O       12.59900000     9.54200000    -2.81700000
N       12.46000000     7.35400000    -3.35700000
H       11.86100000     6.53400000    -3.41500000
H       14.48900000     7.68800000    -2.99900000
H       10.06600000     7.46700000    -2.60500000
H       14.13178000     6.19365900    -3.91018500
H       14.04924000     7.81293900    -4.69425300
H       10.17089000     9.01371600    -1.72630000
H        9.87438900     9.03877200    -3.43124000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '157LYS-160VAL-1')] = qcdb.Molecule("""
0 1
C        3.73200000     8.86900000     8.77600000
C        5.41400000    11.67200000     6.81700000
C        4.49300000    10.13500000     8.44200000
O        5.07500000    10.73200000     9.33700000
N        4.55300000    10.54300000     7.17100000
H        3.43200000     8.95100000     9.82500000
H        4.08200000    10.01800000     6.44700000
H        6.35000000    11.53500000     7.36400000
H        2.84034700     8.73392400     8.14615100
H        4.40318400     8.00483200     8.66319000
H        5.71915900    11.69125000     5.76035100
H        5.02919900    12.64014000     7.17003500
--
0 1
C        4.97300000    13.32400000    11.83600000
C        8.48500000    13.69100000    13.21600000
C        6.48000000    13.17800000    11.91200000
O        7.07600000    12.38500000    11.18200000
N        7.09100000    13.89700000    12.85200000
H        4.68700000    14.23400000    12.35700000
H        6.55200000    14.58800000    13.36600000
H        8.86800000    12.79900000    12.72700000
H        4.63646700    13.38184000    10.79034000
H        4.50454700    12.47876000    12.36148000
H        9.10910400    14.52937000    12.87303000
H        8.55369600    13.58398000    14.30862000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '162ALA-176GLY-1')] = qcdb.Molecule("""
0 1
C      -17.74000000    15.19200000    -3.17800000
C      -19.41700000    14.10800000    -6.43200000
C      -18.74600000    14.52100000    -4.10900000
O      -19.69500000    13.88200000    -3.65200000
N      -18.52100000    14.65400000    -5.41600000
H      -17.74200000    14.64900000    -2.23200000
H      -17.71400000    15.18400000    -5.72800000
H      -20.44200000    14.37700000    -6.19400000
H      -16.72386000    15.16183000    -3.59818400
H      -18.04092000    16.23043000    -2.97522200
H      -19.16579000    14.51369000    -7.42311700
H      -19.37444000    13.01229000    -6.51928100
--
0 1
C      -22.69100000    11.53800000    -5.04700000
C      -22.98100000    14.60100000    -2.82700000
C      -23.41300000    13.34400000    -3.56700000
O      -24.58600000    12.97500000    -3.58500000
N      -22.43900000    12.70700000    -4.21300000
H      -21.49200000    13.06500000    -4.13100000
H      -21.75700000    10.99000000    -5.17000000
H      -23.40600000    10.88100000    -4.54900000
H      -21.89300000    14.64500000    -2.88500000
H      -23.08232000    11.77048000    -6.04840900
H      -23.36436000    15.52015000    -3.29412100
H      -23.21900000    14.58874000    -1.75312500
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '167GLY-232GLY-1')] = qcdb.Molecule("""
0 1
C      -21.57600000     9.57000000    -2.09200000
C      -20.37700000     7.34000000     0.75300000
C      -21.61500000     8.53700000    -0.97500000
O      -22.67400000     8.01200000    -0.64600000
N      -20.45500000     8.23400000    -0.39400000
H      -22.44500000    10.20800000    -2.05100000
H      -21.61900000     9.02700000    -3.03400000
H      -19.59500000     8.65100000    -0.73300000
H      -20.89200000     6.40600000     0.52200000
H      -20.66074000    10.18017000    -2.09124500
H      -19.32348000     7.11306700     0.97346980
H      -20.86994000     7.79696100     1.62374400
--
0 1
C      -25.40700000     7.36400000     1.57700000
C      -24.12100000     4.80400000    -0.91800000
C      -24.99400000     5.41800000     0.16600000
O      -26.03300000     4.87100000     0.52800000
N      -24.57900000     6.57100000     0.67700000
H      -23.73400000     6.99300000     0.30800000
H      -24.86300000     8.27300000     1.83500000
H      -26.32700000     7.64800000     1.06300000
H      -23.18500000     5.36400000    -0.95400000
H      -25.67084000     6.85081200     2.51349600
H      -23.90490000     3.74081100    -0.73653500
H      -24.61164000     4.94192400    -1.89280900
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '168GLY-172LYS-2')] = qcdb.Molecule("""
0 1
C       29.97500000    85.51300000    43.09100000
C       28.46200000    89.04000000    43.07000000
C       29.20500000    86.74000000    43.54900000
O       28.70500000    86.74100000    44.67100000
N       29.09600000    87.76800000    42.69800000
H       30.09400000    84.84500000    43.94300000
H       30.95700000    85.82400000    42.74100000
H       29.51400000    87.69400000    41.77500000
H       29.03000000    89.47400000    43.89100000
H       29.49880000    84.93974000    42.28192000
H       28.46200000    89.74031000    42.22173000
H       27.43080000    88.94035000    43.43973000
--
0 1
C       27.48200000    87.78600000    48.05100000
C       24.81700000    85.78900000    46.24600000
C       25.56100000    86.52100000    47.36000000
O       25.06000000    86.62900000    48.49100000
N       26.74200000    87.04900000    47.03900000
H       27.17300000    86.89600000    46.13100000
H       27.44300000    87.23500000    48.98800000
H       23.74900000    85.90300000    46.42100000
H       28.53329000    87.90968000    47.75186000
H       27.02877000    88.75824000    48.29460000
H       25.00271000    86.14087000    45.22047000
H       25.05560000    84.71708000    46.18223000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '171ALA-175GLU-1')] = qcdb.Molecule("""
0 1
C       14.98600000    -3.67500000    22.14700000
C       11.18700000    -3.38600000    22.07700000
C       13.59900000    -3.04200000    22.08200000
O       13.46500000    -1.81600000    22.03600000
N       12.55600000    -3.86700000    22.09600000
H       15.68700000    -2.89400000    22.45000000
H       12.72000000    -4.86800000    22.08900000
H       11.03100000    -2.78000000    21.18400000
H       15.02070000    -4.47983800    22.89602000
H       15.30987000    -4.05657500    21.16746000
H       10.51184000    -4.25334700    22.03381000
H       10.98391000    -2.76218200    22.95995000
--
0 1
C       11.85300000     1.81500000    22.39000000
C       13.73600000     0.17000000    25.31900000
C       13.40300000     1.16000000    24.21100000
O       14.09000000     2.16100000    24.06300000
N       12.33900000     0.90900000    23.44600000
H       11.81400000     0.06000000    23.61600000
H       11.02500000     1.29900000    21.90200000
H       13.58800000     0.69400000    26.26400000
H       11.44898000     2.75557700    22.79260000
H       12.58831000     1.99522200    21.59198000
H       13.09472000    -0.72358300    25.30248000
H       14.78968000    -0.14379990    25.28301000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '172GLY-175TRP-1')] = qcdb.Molecule("""
0 1
C      -28.21300000    14.47800000    -2.71000000
C      -27.17200000    17.90900000    -3.95200000
C      -27.38700000    15.73300000    -2.89500000
O      -26.25800000    15.81800000    -2.40600000
N      -27.94100000    16.72000000    -3.59600000
H      -28.85200000    14.31300000    -3.57700000
H      -28.84900000    14.61400000    -1.83400000
H      -28.87700000    16.61500000    -3.97600000
H      -27.76600000    18.53800000    -4.61600000
H      -27.55033000    13.61155000    -2.56810700
H      -26.93740000    18.49120000    -3.04866600
H      -26.27170000    17.58541000    -4.49491100
--
0 1
C      -22.98100000    14.60100000    -2.82700000
C      -23.52700000    17.62200000    -5.09600000
C      -23.03600000    16.25600000    -4.61900000
O      -22.24200000    15.61000000    -5.30200000
N      -23.48900000    15.81900000    -3.44600000
H      -24.22100000    16.32600000    -2.96900000
H      -21.89300000    14.64500000    -2.88500000
H      -23.72300000    17.53900000    -6.16600000
H      -23.21900000    14.58874000    -1.75312500
H      -23.29337000    13.69209000    -3.36207800
H      -24.45293000    17.94165000    -4.59551800
H      -22.70930000    18.34757000    -4.97388100
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '197TYR-201LEU-2')] = qcdb.Molecule("""
0 1
C       19.26600000    20.16900000     9.99800000
C       21.03600000    22.26700000     7.30200000
C       19.78500000    20.62600000     8.61400000
O       19.57500000    19.93600000     7.60900000
N       20.48000000    21.76100000     8.57600000
H       18.96400000    19.12600000     9.91300000
H       20.59100000    22.31400000     9.42400000
H       20.22800000    22.44100000     6.59000000
H       20.06652000    20.24370000    10.74872000
H       18.39923000    20.77160000    10.30715000
H       21.60486000    23.19755000     7.44510600
H       21.69911000    21.51272000     6.85327900
--
0 1
C       19.44300000    18.91200000     3.81900000
C       20.93200000    16.82600000     6.68300000
C       20.31100000    17.16100000     5.33000000
O       20.07600000    16.23800000     4.52300000
N       20.05100000    18.46900000     5.08600000
H       20.17800000    19.14200000     5.83600000
H       18.71800000    18.17400000     3.47600000
H       21.53300000    15.92100000     6.56900000
H       18.95578000    19.89001000     3.94597700
H       20.21336000    18.98406000     3.03711000
H       21.57929000    17.64764000     7.02348000
H       20.16373000    16.64485000     7.44912300
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '199SER-202TYR-1')] = qcdb.Molecule("""
0 1
C       43.86400000    26.88100000    10.53600000
C       43.20800000    24.30000000     7.80900000
C       43.40000000    26.28500000     9.20200000
O       42.94600000    27.00200000     8.31400000
N       43.51200000    24.96600000     9.08500000
H       43.49500000    27.90700000    10.56100000
H       43.90400000    24.43600000     9.85400000
H       43.79600000    24.75100000     7.00300000
H       43.49007000    26.34465000    11.42059000
H       44.96194000    26.92526000    10.58669000
H       43.46626000    23.23348000     7.88538800
H       42.15043000    24.41349000     7.52852400
--
0 1
C       39.44100000    28.17900000     8.36500000
C       39.42900000    24.37700000     8.11400000
C       39.06400000    25.83200000     7.84100000
O       38.24000000    26.11900000     6.98200000
N       39.67800000    26.76100000     8.58100000
H       40.30700000    26.48000000     9.32800000
H       38.37700000    28.38800000     8.48700000
H       39.15600000    23.79600000     7.22900000
H       40.00118000    28.77311000     9.10204200
H       39.72832000    28.49440000     7.35111100
H       40.50351000    24.25223000     8.31362800
H       38.85022000    23.99139000     8.96624700
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '205THR-208GLU-1')] = qcdb.Molecule("""
0 1
C       27.52100000    12.50600000    26.48200000
C       29.67700000    10.31900000    28.70500000
C       28.05900000    11.83100000    27.72700000
O       27.63600000    12.13900000    28.84700000
N       29.02300000    10.93300000    27.56400000
H       26.99500000    13.40000000    26.81100000
H       29.34100000    10.70300000    26.62900000
H       28.94100000     9.74600000    29.27000000
H       28.34116000    12.82075000    25.81997000
H       26.77913000    11.87706000    25.96813000
H       30.48129000     9.63955500    28.38644000
H       30.06111000    11.10674000    29.36977000
--
0 1
C       28.58400000    15.47400000    30.65900000
C       31.59300000    13.31500000    29.94300000
C       30.60700000    14.23600000    30.64100000
O       30.85100000    14.63700000    31.78200000
N       29.50600000    14.58700000    29.99000000
H       29.33700000    14.29100000    29.03500000
H       29.17400000    16.23700000    31.17100000
H       32.12100000    12.77800000    30.73300000
H       28.00426000    16.04422000    29.91822000
H       28.00915000    14.98034000    31.45640000
H       31.10898000    12.57904000    29.28415000
H       32.32319000    13.91761000    29.38292000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '205THR-209LEU-2')] = qcdb.Molecule("""
0 1
C       27.52100000    12.50600000    26.48200000
C       29.67700000    10.31900000    28.70500000
C       28.05900000    11.83100000    27.72700000
O       27.63600000    12.13900000    28.84700000
N       29.02300000    10.93300000    27.56400000
H       26.99500000    13.40000000    26.81100000
H       29.34100000    10.70300000    26.62900000
H       28.94100000     9.74600000    29.27000000
H       28.34116000    12.82075000    25.81997000
H       26.77913000    11.87706000    25.96813000
H       30.48129000     9.63955500    28.38644000
H       30.06111000    11.10674000    29.36977000
--
0 1
C       26.86500000    12.72100000    32.66600000
C       28.58400000    15.47400000    30.65900000
C       27.79100000    14.79300000    31.75900000
O       27.31800000    15.47600000    32.67200000
N       27.66900000    13.46700000    31.70400000
H       28.08700000    12.97100000    30.92600000
H       26.13800000    13.39700000    33.11100000
H       29.17400000    16.23700000    31.17100000
H       26.30931000    11.90977000    32.17293000
H       27.42080000    12.32765000    33.52992000
H       29.28648000    14.79819000    30.14929000
H       28.00426000    16.04422000    29.91822000
units angstrom
""")

GEOS['%s-%s-dimer' % (dbse, '228ARG-232GLU-1')] = qcdb.Molecule("""
0 1
C       48.27300000    30.47900000    27.65500000
C       46.29600000    28.47400000    25.10100000
C       47.53200000    30.13600000    26.37000000
O       47.41000000    30.96700000    25.48200000
N       47.04600000    28.89700000    26.27700000
H       48.24600000    31.56700000    27.76600000
H       47.26200000    28.22400000    27.00400000
H       46.93800000    28.52200000    24.22000000
H       47.79481000    30.00687000    28.52588000
H       49.32228000    30.15311000    27.60200000
H       45.95848000    27.43677000    25.24327000
H       45.42364000    29.12286000    24.93374000
--
0 1
C       46.62400000    32.86600000    22.40200000
C       45.05200000    33.85700000    25.70700000
C       45.40700000    33.90800000    24.22600000
O       44.98000000    34.79300000    23.48100000
N       46.20200000    32.93500000    23.79700000
H       46.52900000    32.22200000    24.44300000
H       47.02700000    33.83900000    22.11300000
H       44.24900000    34.57300000    25.88800000
H       47.42942000    32.12782000    22.27396000
H       45.78943000    32.67093000    21.71248000
H       44.71165000    32.84806000    25.98306000
H       45.91216000    34.12701000    26.33726000
units angstrom
""")

# <<< Derived Geometry Strings >>>
for rxn in HRXN:
    GEOS['%s-%s-monoA-unCP' % (dbse, rxn)] = GEOS['%s-%s-dimer' % (dbse, rxn)].extract_fragments(1)
    GEOS['%s-%s-monoB-unCP' % (dbse, rxn)] = GEOS['%s-%s-dimer' % (dbse, rxn)].extract_fragments(2)
    GEOS['%s-%s-monoA-CP'   % (dbse, rxn)] = GEOS['%s-%s-dimer' % (dbse, rxn)].extract_fragments(1, 2)
    GEOS['%s-%s-monoB-CP'   % (dbse, rxn)] = GEOS['%s-%s-dimer' % (dbse, rxn)].extract_fragments(2, 1)

