/*	Conductor_Matrix_Model

PIRL CVS ID: Conductor_Matrix_Model.java,v 1.15 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor.Maestro;

import	PIRL.Messenger.Message;

import	javax.swing.table.AbstractTableModel;
import	javax.swing.event.TableModelListener;
import	javax.swing.event.TableModelEvent;
import	java.util.Vector;


/**	A <i>Conductor_Matrix_Model</i> contains the data for a matrix
	of the number of Conductors at each Theater location.
<p>
	Each column of the matrix table is for a theater location. Each row
	is for a Conductor name. Each cell contains the number of Conductors
	having the row name operating on the theater location of the column
	name.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.15
*/
public class Conductor_Matrix_Model
	extends AbstractTableModel
	implements TableModelListener
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.Conductor_Matrix_Model (1.15 2012/04/16 06:04:11)";


public static final String
	CONDUCTORS_COLUMN_NAME		= "Conductors",
	TOTALS_NAME					= "Totals";

public static final int
	RUN_TO_WAIT_STATE			= Conductor_Table.RUN_TO_WAIT_STATE,
	RUNNING_STATE				= Conductor_Table.RUNNING_STATE,
	POLLING_STATE				= Conductor_Table.POLLING_STATE,
	WAITING_STATE				= Conductor_Table.WAITING_STATE,
	HALTED_STATE				= Conductor_Table.HALTED_STATE;

private Vector<String>
	Theater_Names				= new Vector<String> (),
	Theater_Locations			= new Vector<String> (),
	Conductor_Names				= new Vector<String> ();
private Vector<Vector<Count>>
	Counts						= new Vector<Vector<Count>> ();


private static final String
	NL							= System.getProperty ("line.separator");


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTOR	= 1 << 0,
	DEBUG_ADD			= 1 << 1,
	DEBUG_REMOVE		= 1 << 2,
	DEBUG_INCREMENTORS	= 1 << 3,
	DEBUG_REBUILD		= 1 << 4,
	DEBUG_LISTENER		= 1 << 5,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
public Conductor_Matrix_Model
	(
	Conductor_Table_Model	table_model
	)
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println (">-< Conductor_Matrix_Model");
table_model.Add_State_Change_Listener (this);
}

/*==============================================================================
	AbstractTableModel
*/
public int findColumn
	(
	String	theater_location
	)
{
int
	column;
if ((column = Theater_Names.indexOf (theater_location)) >= 0 ||
	(column = Theater_Locations.indexOf (theater_location)) >= 0)
	column++;;
return column;
}


public int findRow
	(
	String	conductor_name
	)
{return Conductor_Names.indexOf (conductor_name);}

/*==============================================================================
	TableModel
*/
public Class getColumnClass
	(
	int		column
	)
{
if (column == 0)
	return String.class;
else
if (column <= Theater_Locations.size ())
	return Count.class;
return Object.class;
}


public int getColumnCount ()
{return Theater_Locations.size () + 1;}


public String getColumnName
	(
	int		column
	)
{
if (column == 0)
	return CONDUCTORS_COLUMN_NAME;
if (--column < Theater_Names.size ())
	return Theater_Names.get (column);
return null;
}


public int getRowCount ()
{return Conductor_Names.size ();}


public Object getValueAt
	(
	int		row,
	int		column
	)
{
try
	{
	if (column == 0)
		return Conductor_Names.get (row);
	else
		return Counts.get (row).get (--column);
	}
catch (IndexOutOfBoundsException exception) {}
return null;
}

/*==============================================================================
	Accessors
*/
/**	Get the table column index for a Conductor identity.
<p>
	@return	The table column index for the Conductor identity. This will
		be -1 if the identity is null, empty, or has a {@link
		Conductor_Table_Model#Theater_Location(Message) Theater location}
		not present in this table.
*/
public int Column
	(
	Message	identity
	)
{return findColumn (Conductor_Table_Model.Theater_Location (identity));}

/**	Get the table row index for a Conductor identity.
<p>
	@return	The table row index for the Conductor identity. This will
		be -1 if the identity is null, empty, or has a {@link
		Conductor_Table_Model#Conductor_Name(Message) Conductor name}
		not present in this table.
*/
public int Row
	(
	Message	identity
	)
{return findRow (Conductor_Table_Model.Conductor_Name (identity));}


public Count Count
	(
	int		row,
	int		column
	)
{
if (column > 0 &&
	column <= getColumnCount () &&
	row >= 0 &&
	row < getRowCount ())
	return Count_at (row, column);
return null;
}


public Count Count
	(
	Message	identity
	)
{
int
	column = Column (identity);
if (column > 0)
	{
	int
		row = Row (identity);
	if (row >= 0)
		return Count_at (row, column);
	}
return null;
}


private Count Count_at
	(
	int		row,
	int		column
	)
{return Counts.get (row).get (--column);}


public int Total
	(
	int		row,
	int		column
	)
{
if (column > 0 &&
	column <= getColumnCount () &&
	row >= 0 &&
	row < getRowCount ())
	return Count_at (row, column).Counts[TOTAL];
return -1;
}


public int Total
	(
	String	theater_location,
	String	conductor_name
	)
{
int
	column = findColumn (conductor_name);
if (column > 0)
	{
	int
		row = findRow (theater_location);
	if (row >= 0)
		return Count_at (row, column).Counts[TOTAL];
	}
return -1;
}


public int Total
	(
	Message	identity
	)
{
int
	column = Column (identity);
if (column > 0)
	{
	int
		row = Row (identity);
	if (row >= 0)
	return Count_at (row, column).Counts[TOTAL];
	}
return -1;
}


protected Conductor_Matrix_Model Add
	(
	Message	identity
	)
{
if ((DEBUG & DEBUG_ADD) != 0)
	System.out.println
		(">>> Conductor_Matrix_Model.Add -" + identity);
if (identity != null)
	{
	String
		theater_location,
		conductor_name;
	if ((theater_location = Conductor_Table_Model.Theater_Location (identity))
			!= null &&
		(conductor_name   = Conductor_Table_Model.Conductor_Name (identity))
			!= null)
		{
		int
			column = findColumn (theater_location),
			row    = findRow (conductor_name);
		Add_to (column, theater_location, row, conductor_name,
				Conductor_Table_Model.Processing_State (identity));

		//	Update the table view.
		if (column < 0)
			fireTableStructureChanged ();
		else
		if (row < 0)
			fireTableRowsInserted (row, row);
		else
			fireTableCellUpdated (row, column);
		}
	}
if ((DEBUG & DEBUG_ADD) != 0)
	System.out.println
		("<<< Conductor_Matrix_Model.Add");
return this;
}


protected void Add_to
	(
	int		column,
	String	theater_location,
	int		row,
	String	conductor_name,
	int		processing_state
	)
{
if ((DEBUG & DEBUG_ADD) != 0)
	System.out.println
		(">>> Add_to:" + NL
		+"    Column " + column + " - " + theater_location + NL
		+"       Row " + row + " - " + conductor_name + NL
		+"    State = " + processing_state);
boolean
	changed = false;

if (column < 0)
	{
	//	Add the new column.
	column = getColumnCount ();
	if ((DEBUG & DEBUG_ADD) != 0)
		System.out.println
			("    Adding column " + column);
	for (int
			rows = getRowCount (),
			index = 0;
			index < rows;
			index++)
		Counts.get (index).add (new Count ());

	changed = Add_to_Theater_Names (theater_location);
	}
if (row < 0)
	{
	//	Add the new row.
	row = getRowCount ();
	if ((DEBUG & DEBUG_ADD) != 0)
		System.out.println
			("    Adding row " + row);
	Vector<Count>
		row_vector = new Vector<Count> (Theater_Locations.size ());
	for (int
			columns = getColumnCount () - 1,
			index = 0;
			index < columns;
			index++)
		row_vector.add (new Count ());
	Counts.add (row_vector);

	Conductor_Names.add (conductor_name);
	}

//	Increment the count.
if ((DEBUG & DEBUG_ADD) != 0)
	System.out.println
		("    Increment count at row " + row + ", column " + column);
Increment (Count_at (row, column), processing_state);
if ((DEBUG & DEBUG_ADD) != 0)
	System.out.println
		("<<< Add_to");
}


private boolean Add_to_Theater_Names
	(
	String	theater_location
	)
{
Theater_Locations.add (Theater.Full_Location (theater_location));
String
	name = Theater.Location (theater_location);

//  Check for duplicate Theater_Names entry.
boolean
	changed = false;
int
	index = Theater_Names.size ();
while (--index >= 0)
	{
	if (Theater_Names.get (index).equals (name))
		{
		name = Theater_Locations.lastElement ();
		Theater_Names.set (index, name);
		changed = true;
		break;
		}
	}
Theater_Names.add (name);

return changed;
}


protected void Rebuild
	(
	Conductor_Table_Model	model
	)
{
if (model == null)
	return;
if ((DEBUG & DEBUG_ADD) != 0)
	System.out.println
		(">>> Conductor_Matrix_Model.Rebuild");

Theater_Names.clear ();
Theater_Locations.clear ();
Conductor_Names.clear ();
Counts.clear ();

int
	model_row = 0;
Message
	identity;
while ((identity = model.Identity (model_row++)) != null)
	{
	String
		theater_location,
		conductor_name;
	if ((theater_location = Conductor_Table_Model.Theater_Location (identity))
			!= null &&
		(conductor_name   = Conductor_Table_Model.Conductor_Name (identity))
			!= null)
		Add_to
			(
			findColumn (theater_location),
			theater_location,
			findRow (conductor_name),
			conductor_name,
			Conductor_Table_Model.Processing_State (identity)
			);
	}

//	Update the table view.
fireTableStructureChanged ();
if ((DEBUG & DEBUG_ADD) != 0)
	System.out.println
		("<<< Conductor_Matrix_Model.Rebuild");
}

/**	Adjust the model based on the removal of a Conductor identity.
<p>
	If the identity can be associated with a table cell its
	Count values are {@link #Decrement(Count, Message) decremented}.
<p>
	@param	identity	A Conductor identity Message.
*/
protected void Removing
	(
	Message	identity
	)
{
if ((DEBUG & DEBUG_REMOVE) != 0)
	System.out.println
		(">>> Conductor_Matrix_Model.Remove -" + identity);
int
	row,
	column;
if ((column = Column (identity)) > 0 &&
	(row = Row (identity)) >= 0)
	{
	if ((DEBUG & DEBUG_REMOVE) != 0)
		System.out.println
			("    row " + row + ", column " + column);
	Decrement (Counts.get (row).get (--column), identity);

	//!!!	Should check for an empty row or column and adjust accordingly.

	fireTableCellUpdated (row, column);
	}
if ((DEBUG & DEBUG_REMOVE) != 0)
	System.out.println
		("<<< Conductor_Matrix_Model.Remove");
}

/*==============================================================================
	Count
*/
/**	Index for arrays of processing state information.
*/
public static final int
	RUNNING		= 0,
	POLLING		= 1,
	WAITING		= 2,
	HALTED		= 3,
	TOTAL		= 4,	//	Must be last in the sequence.
	UNKNOWN		= 5;	//	Must be greater than TOTAL.

/**	A <i>Count</i> contains table cell Conductor processing state count values.
<p>
	A count of the number of Conductors in each processing state,
	including the unknown state, is maintained plus the total number of
	Conductors.
*/
public class Count
{
/**	The Conductor processing state counts.
*/
public int
	Counts[] = {0, 0, 0, 0, 0, 0};


public String toString ()
{return String.valueOf (Counts[TOTAL]);}
}


private void Increment
	(
	Count	count,
	Message	identity
	)
{Increment (count, Conductor_Table_Model.Processing_State (identity));}


private void Increment
	(
	Count	count,
	int		processing_state
	)
{
if (count == null)
	return;

count.Counts[TOTAL]++;
switch (processing_state)
	{
	case RUN_TO_WAIT_STATE:
	case RUNNING_STATE:	count.Counts[RUNNING]++; break;
	case POLLING_STATE:	count.Counts[POLLING]++; break;
	case WAITING_STATE:	count.Counts[WAITING]++; break;
	case HALTED_STATE:	count.Counts[HALTED]++; break;
	default:			count.Counts[UNKNOWN]++;
	}
if ((DEBUG & DEBUG_INCREMENTORS) != 0)
	System.out.println
		(">-< Increment: "
			+ count.Counts[RUNNING] + "R "
			+ count.Counts[POLLING] + "P "
			+ count.Counts[WAITING] + "W "
			+ count.Counts[HALTED]  + "H "
			+ count.Counts[UNKNOWN] + "U "
			+ count.Counts[TOTAL] + "T");
}


private void Decrement
	(
	Count	count,
	Message	identity
	)
{Decrement (count, Conductor_Table_Model.Processing_State (identity));}


private void Decrement
	(
	Count	count,
	int		processing_state
	)
{
if (count == null)
	return;

count.Counts[TOTAL]--;
switch (processing_state)
	{
	case RUN_TO_WAIT_STATE:
	case RUNNING_STATE:	count.Counts[RUNNING]--; break;
	case POLLING_STATE:	count.Counts[POLLING]--; break;
	case WAITING_STATE:	count.Counts[WAITING]--; break;
	case HALTED_STATE:	count.Counts[HALTED]--; break;
	default:			count.Counts[UNKNOWN]--;
	}
if ((DEBUG & DEBUG_INCREMENTORS) != 0)
	System.out.println
		(">-< Decrement: "
			+ count.Counts[RUNNING] + "R "
			+ count.Counts[POLLING] + "P "
			+ count.Counts[WAITING] + "W "
			+ count.Counts[HALTED]  + "H "
			+ count.Counts[UNKNOWN] + "U "
			+ count.Counts[TOTAL] + "T");
}


private void Adjust
	(
	Count	count,
	Message	identity,
	int		previous_state
	)
{
int
	processing_state = Conductor_Table_Model.Processing_State (identity);
if (processing_state != previous_state);
	{
	Increment (count, processing_state);
	Decrement (count, previous_state);
	}
}

/*==============================================================================
	TableModelListener
*/
public void tableChanged
	(
	TableModelEvent	event
	)
{
if (! (event instanceof Conductor_Table_Model_Event))
	return;
if ((DEBUG & DEBUG_LISTENER) != 0)
	System.out.println
		(">>> Conductor_Matrix_Model.tableChanged");

Conductor_Table_Model
	model = (Conductor_Table_Model)event.getSource ();

switch (event.getType ())
	{
	case TableModelEvent.INSERT:
		if ((DEBUG & DEBUG_LISTENER) != 0)
			System.out.println
				("    INSERT: from "
					+ event.getFirstRow () + " to "
					+ event.getLastRow ());
		for (int
				index = event.getFirstRow ();
				index <= event.getLastRow ();
				index++)
			Add (model.Identity (index));
		break;

	case TableModelEvent.UPDATE:
		if ((DEBUG & DEBUG_LISTENER) != 0)
			System.out.println
				("    UPDATE: from "
					+ event.getFirstRow () + " to "
					+ event.getLastRow ());
		if (event.getFirstRow () == TableModelEvent.HEADER_ROW ||
			event.getLastRow () > getRowCount ())
			/*
				The changes to the source data model are non-specific.
				Since source rows may have been added or removed the
				entire matrix must be rebuilt.
			*/
			Rebuild (model);
		else
			{
			Message
				identity;
			int
				row,
				column;
			Count
				count;
			for (int
					index = event.getFirstRow ();
					index <= event.getLastRow ();
					index++)
				{
				if ((identity = model.Identity (index)) != null &&
					(column = Column (identity)) > 0 &&
					(row = Row (identity)) >= 0 &&
					(count = Count_at (row, column)) != null)
					{
					Adjust (count, identity,
						((Conductor_Table_Model_Event)event).Processing_State ());
					fireTableCellUpdated (row, column);
					}
				}
			}
		break;

	case TableModelEvent.DELETE:
		if ((DEBUG & DEBUG_LISTENER) != 0)
			System.out.println
				("    DELETE: from "
					+ event.getFirstRow () + " to "
					+ event.getLastRow ());
		for (int
				index = event.getFirstRow ();
				index <= event.getLastRow ();
				index++)
			Removing (model.Identity (index));
		break;
	}
if ((DEBUG & DEBUG_LISTENER) != 0)
	System.out.println
		("<<< Conductor_Matrix_Model.tableChanged");
}



}
