/*
 * Copyright (c) 2013 Intel Corporation
 * Copyright (c) 2016 Advanced Micro Devices
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 *
 */

/* Test indirect data generated by transform feedback to catch flushing bugs.
 *
 * Based on draw-arrays.c.
 */

#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_core_version = 31;

	config.window_visual = PIGLIT_GL_VISUAL_DOUBLE | PIGLIT_GL_VISUAL_RGB;
	config.khr_no_error_support = PIGLIT_NO_ERRORS;

PIGLIT_GL_TEST_CONFIG_END

GLuint draw_vao;
GLint tf_prog, draw_prog;
GLint tf_in;

float red[] = {1,0,0};
float blue[] = {0,0,1};

float vertices_data[] = {
	-1, -1,
	 1, -1,
	-1,  1,
};

GLint indirect_data[] = {
	3,		/* count */
	1,		/* primcount */
	0,		/* first vertex */
	0,		/* mbz */
};

enum piglit_result
piglit_display(void)
{
	bool pass = true;

	glViewport(0, 0, 128, 128);

	glClearColor(0,0,1,1);
	glClear(GL_COLOR_BUFFER_BIT);

	/* Set vertex array already for transform feedback - its contents are
	 * irrelevant. */
	glBindVertexArray(draw_vao);

	glUseProgram(tf_prog);
	glUniform4iv(tf_in, 1, indirect_data);

	glEnable(GL_RASTERIZER_DISCARD);
	glBeginTransformFeedback(GL_POINTS);
	glDrawArrays(GL_POINTS, 0, 1);
	glEndTransformFeedback();
	glDisable(GL_RASTERIZER_DISCARD);

	glUseProgram(draw_prog);

	glDrawArraysIndirect(GL_TRIANGLES, (GLvoid const *)0);

	glUseProgram(0);

	piglit_present_results();

	pass = piglit_probe_pixel_rgb(32, 32, red) && pass;
	pass = piglit_probe_pixel_rgb(96, 96, blue) && pass;

	return pass ? PIGLIT_PASS : PIGLIT_FAIL;
}

void
piglit_init(int argc, char **argv)
{
	static const char* const tf_out = "tf_out";
	GLint zeros[4] = { 0, 0, 0, 0 };
	GLuint vertices_bo;
	GLuint indirect_bo;

	piglit_require_extension("GL_ARB_draw_indirect");

	glGenVertexArrays(1, &draw_vao);
	glBindVertexArray(draw_vao);

	glGenBuffers(1, &vertices_bo);
	glBindBuffer(GL_ARRAY_BUFFER, vertices_bo);
	glBufferData(GL_ARRAY_BUFFER, sizeof(vertices_data), vertices_data, GL_STATIC_DRAW);
	glEnableVertexAttribArray(0);
	glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 0, 0);

	glGenBuffers(1, &indirect_bo);
	glBindBuffer(GL_DRAW_INDIRECT_BUFFER, indirect_bo);
	/* Fill with zeros so that the default contents are safe (don't lead
	 * to excessive GPU processing times that will be mistaken for hangs).
	 */
	glBufferData(GL_DRAW_INDIRECT_BUFFER, sizeof(indirect_data), zeros, GL_STATIC_DRAW);

	draw_prog = piglit_build_simple_program(
			"#version 130\n"
			"#extension GL_ARB_explicit_attrib_location: require\n"
			"\n"
			"layout(location=0) in vec2 pos;\n"
			"\n"
			"void main() {\n"
			"	gl_Position = vec4(pos, 0, 1);\n"
			"}\n",

			"#version 130\n"
			"\n"
			"void main() {\n"
			"	gl_FragColor = vec4(1,0,0,1);\n"
			"}\n");

	glBindVertexArray(0);

	tf_prog = piglit_build_simple_program_unlinked(
			"#version 130\n"
			"\n"
			"out ivec4 tf_out;\n"
			"uniform ivec4 tf_in;\n"
			"\n"
			"void main() {\n"
			"	tf_out = tf_in;\n"
			"	gl_Position = vec4(0);\n"
			"}\n",
			NULL);
	glTransformFeedbackVaryings(tf_prog, 1, &tf_out, GL_INTERLEAVED_ATTRIBS);
	glLinkProgram(tf_prog);
	if (!piglit_link_check_status(tf_prog))
		piglit_report_result(PIGLIT_FAIL);
	piglit_check_gl_error(GL_NO_ERROR);

	tf_in = glGetUniformLocation(tf_prog, "tf_in");

	glBindBufferRange(GL_TRANSFORM_FEEDBACK_BUFFER, 0, indirect_bo, 0, sizeof(indirect_data));
}
