
package pauker.program.gui.swing;

import java.awt.Dimension;
import java.util.List;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.JTextComponent;
import pauker.program.Batch;
import pauker.program.Card;
import pauker.program.CardSide;
import pauker.program.SearchEngine;

/**
 * The dialog used when adding new cards to a lesson
 * @author ronny
 */
public class NewCardDialog extends JDialog {

    private final PaukerFrame paukerFrame;
    private final BatchListModel similarCardsListModel;
    private final BatchListCellRenderer batchListCellRenderer;
    private final JTextComponent frontSideTextComponent;
    private final JTextComponent reverseSideTextComponent;
    private final SearchEngine searchEngine;
    private boolean showSimilarCards;

    /** Creates new form AddCardDialog
     * @param paukerFrame the pauker frame
     * @param searchEngine the engine for searching similar cards
     * @param keepOpen if the dialog should be kept open when pressing the
     * OK button
     */
    public NewCardDialog(PaukerFrame paukerFrame,
            SearchEngine searchEngine, boolean keepOpen) {
        super(paukerFrame, true);
        this.paukerFrame = paukerFrame;
        this.searchEngine = searchEngine;

        setTitle(PaukerFrame.STRINGS.getString("Add_Card"));

        initComponents();

        editCardPanel.setPaukerFrame(paukerFrame);
        editCardPanel.setSplitOrientation(paukerFrame.getSplitOrientation());
        editCardPanel.applyWrapSettings();
        editCardPanel.setNextFocusComponent(okButton);
        frontSideTextComponent = editCardPanel.getFrontSideTextComponent();
        reverseSideTextComponent = editCardPanel.getReverseSideTextComponent();

        showSimilarCards = true;

        // add a listener to the editCardPanel textAreas
        // used for the preview of similar or existing cards
        frontSideTextComponent.getDocument().addDocumentListener(
                new MyDocumentListener(Card.Element.FRONT_SIDE));
        reverseSideTextComponent.getDocument().addDocumentListener(
                new MyDocumentListener(Card.Element.REVERSE_SIDE));

        similarCardsListModel = new BatchListModel();
        similarCardsList.setModel(similarCardsListModel);
        batchListCellRenderer = new BatchListCellRenderer(
                paukerFrame, similarCardsList.getSelectionBackground());
        batchListCellRenderer.disableInfoPanel();
        batchListCellRenderer.setLineWrap(paukerFrame.isLineWrapping());
        similarCardsList.setCellRenderer(batchListCellRenderer);
        similarCardsList.setLineWrapping(paukerFrame.isLineWrapping());

        // set keepOpenCheckBox to last saved state
        keepOpenCheckBox.setSelected(keepOpen);

        pack();

        // fix minimum sizes for a good splitpane behaviour
        similarSeparator1.setMinimumSize(similarSeparator1.getPreferredSize());
        similarLabel.setMinimumSize(similarLabel.getPreferredSize());
        similarSeparator2.setMinimumSize(similarSeparator2.getPreferredSize());
        similarCardsPanel.setMinimumSize(similarCardsPanel.getPreferredSize());

        setLocationRelativeTo(paukerFrame);
    }

    /**
     * resets this dialog so that it can be used again
     */
    public void reset() {
        editCardPanel.reset();
        editCardPanel.setNextFocusComponent(okButton);
        similarCardsListModel.clear();
        similarCardsList.setLineWrapping(paukerFrame.isLineWrapping());
        batchListCellRenderer.setLineWrap(paukerFrame.isLineWrapping());
        setLocationRelativeTo(paukerFrame);
    }

    /**
     * sets the orientation of the split pane in the dialog
     * @param orientation the orientation of the split pane in the dialog
     */
    public void setOrientation(int orientation) {
        editCardPanel.setSplitOrientation(orientation);
    }

    /**
     * returns the EditCardPanel used in this dialog
     * @return the EditCardPanel used in this dialog
     */
    public EditCardPanel getEditCardPanel() {
        return editCardPanel;
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        splitPane = new javax.swing.JSplitPane();
        editCardPanel = new pauker.program.gui.swing.EditCardPanel();
        similarCardsPanel = new javax.swing.JPanel();
        similarSeparator1 = new javax.swing.JSeparator();
        similarLabel = new javax.swing.JLabel();
        similarSeparator2 = new javax.swing.JSeparator();
        batchListScrollPane = new javax.swing.JScrollPane();
        similarCardsList = new pauker.program.gui.swing.UpdateLayoutList();
        buttonPanel = new javax.swing.JPanel();
        keepOpenCheckBox = new javax.swing.JCheckBox();
        okButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();

        getContentPane().setLayout(new java.awt.GridBagLayout());

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
        });
        addComponentListener(new java.awt.event.ComponentAdapter() {
            public void componentShown(java.awt.event.ComponentEvent evt) {
                formComponentShown(evt);
            }
        });

        splitPane.setOrientation(javax.swing.JSplitPane.VERTICAL_SPLIT);
        splitPane.setResizeWeight(0.5);
        splitPane.setContinuousLayout(true);
        splitPane.setOneTouchExpandable(true);
        splitPane.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
            public void propertyChange(java.beans.PropertyChangeEvent evt) {
                splitPanePropertyChange(evt);
            }
        });

        editCardPanel.setRepeatByTyping(false);
        splitPane.setTopComponent(editCardPanel);

        similarCardsPanel.setLayout(new java.awt.GridBagLayout());

        similarSeparator1.setPreferredSize(new java.awt.Dimension(20, 2));
        similarCardsPanel.add(similarSeparator1, new java.awt.GridBagConstraints());

        similarLabel.setFont(new java.awt.Font("Dialog", 0, 10));
        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("pauker/Strings"); // NOI18N
        similarLabel.setText(bundle.getString("Similar_Cards")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        similarCardsPanel.add(similarLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        similarCardsPanel.add(similarSeparator2, gridBagConstraints);

        similarCardsList.setName("similarCardsList");
        similarCardsList.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                similarCardsListMouseClicked(evt);
            }
        });

        batchListScrollPane.setViewportView(similarCardsList);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        similarCardsPanel.add(batchListScrollPane, gridBagConstraints);

        splitPane.setBottomComponent(similarCardsPanel);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        getContentPane().add(splitPane, gridBagConstraints);

        buttonPanel.setLayout(new java.awt.GridBagLayout());

        keepOpenCheckBox.setFont(new java.awt.Font("Dialog", 0, 10));
        keepOpenCheckBox.setText(bundle.getString("Keep_Dialog_Open")); // NOI18N
        keepOpenCheckBox.setName("keepOpenCheckBox");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        buttonPanel.add(keepOpenCheckBox, gridBagConstraints);

        okButton.setFont(new java.awt.Font("Dialog", 0, 10));
        okButton.setText(bundle.getString("OK")); // NOI18N
        okButton.setName("okButton");
        okButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                okButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        buttonPanel.add(okButton, gridBagConstraints);

        cancelButton.setFont(new java.awt.Font("Dialog", 0, 10));
        cancelButton.setText(bundle.getString("Cancel")); // NOI18N
        cancelButton.setName("cancelButton");
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        buttonPanel.add(cancelButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 5, 5);
        getContentPane().add(buttonPanel, gridBagConstraints);

    }// </editor-fold>//GEN-END:initComponents

    private void similarCardsListMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_similarCardsListMouseClicked
        if (evt.getClickCount() == 2) {
            //int selectedIndex = batchList.locationToIndex(evt.getPoint());
            int selectedIndex = similarCardsList.getSelectedIndex();
            if (selectedIndex != -1) {
                Card selectedCard =
                        (Card) similarCardsListModel.get(selectedIndex);
                EditCardDialog editCardDialog = new EditCardDialog(
                        this, paukerFrame, selectedCard, searchEngine);
                if (editCardDialog.okPressed()) {
                    similarCardsListModel.cardsChanged(
                            selectedIndex, selectedIndex);
                    paukerFrame.somethingHasChanged();
                }
            }
        }
    }//GEN-LAST:event_similarCardsListMouseClicked

    private void splitPanePropertyChange(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_splitPanePropertyChange
        // do we hide the similar cards list?
        Dimension splitPaneSize = splitPane.getSize();
        int dividerLocation = splitPane.getDividerLocation();
        int dividerSize = splitPane.getDividerSize();
        // the "< 3" is just a safety guess, the observed value was always
        // "== 1"
        if (splitPaneSize.height - dividerLocation - dividerSize < 3) {
            showSimilarCards = false;
        } else {
            // only switch on search if it was previously switched off
            if (!showSimilarCards) {
                showSimilarCards = true;
                Card.Element lastFocusSide = EditCardPanel.getLastFocusSide();
                if (lastFocusSide == Card.Element.FRONT_SIDE) {
                    searchSimilarCards(editCardPanel.getFrontSide(),
                            SearchEngine.FRONT_SIDE);
                } else {
                    searchSimilarCards(editCardPanel.getReverseSide(),
                            SearchEngine.REVERSE_SIDE);
                }
            }
        }
    }//GEN-LAST:event_splitPanePropertyChange

    private void formComponentShown(java.awt.event.ComponentEvent evt) {//GEN-FIRST:event_formComponentShown
        editCardPanel.setFocus(Card.Element.FRONT_SIDE);
        // Bug#1721717
        // default button is not correctly set
        // !!! setting the default button without SwingUtilities seems not to
        // work with Windows XP and Java 6 !!!
        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                getRootPane().setDefaultButton(okButton);
            }
        });
    }//GEN-LAST:event_formComponentShown

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
        exitDialog();
    }//GEN-LAST:event_cancelButtonActionPerformed

    private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_okButtonActionPerformed

        String frontSideText = editCardPanel.getFrontSide();
        String reverseSideText = editCardPanel.getReverseSide();

        if (frontSideText.length() == 0) {
            JOptionPane.showMessageDialog(this,
                    PaukerFrame.STRINGS.getString("FrontSide_Is_Empty"),
                    PaukerFrame.STRINGS.getString("Error"),
                    JOptionPane.ERROR_MESSAGE);

            editCardPanel.setFocus(Card.Element.FRONT_SIDE);

            return;
        }

        if (reverseSideText.length() == 0) {
            JOptionPane.showMessageDialog(this,
                    PaukerFrame.STRINGS.getString("ReverseSide_Is_Empty"),
                    PaukerFrame.STRINGS.getString("Error"),
                    JOptionPane.ERROR_MESSAGE);

            editCardPanel.setFocus(Card.Element.REVERSE_SIDE);
            return;
        }

        // set front side
        CardSide frontSide = new CardSide(frontSideText);
        frontSide.setFont(editCardPanel.getFrontSideFont());
        frontSide.setForegroundColor(editCardPanel.getFrontForegroundColor());
        frontSide.setBackgroundColor(editCardPanel.getFrontBackgroundColor());
        frontSide.setOrientation(
                editCardPanel.getFrontSideComponentOrientation());
        frontSide.setRepeatByTyping(editCardPanel.isRepeatByTyping());

        // set reverse side
        CardSide reverseSide = new CardSide(reverseSideText);
        reverseSide.setFont(editCardPanel.getReverseSideFont());
        reverseSide.setForegroundColor(
                editCardPanel.getReverseSideForegroundColor());
        reverseSide.setBackgroundColor(
                editCardPanel.getReverseSideBackgroundColor());
        reverseSide.setOrientation(
                editCardPanel.getReverseSideComponentOrientation());

        Card newCard = new Card(frontSide, reverseSide);
        paukerFrame.addCard(newCard);

        if (keepOpenCheckBox.isSelected()) {
            editCardPanel.clear();
            editCardPanel.setFocus(Card.Element.FRONT_SIDE);
            similarCardsListModel.clear();
            similarCardsList.repaint();
        } else {
            exitDialog();
        }
    }//GEN-LAST:event_okButtonActionPerformed

    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_closeDialog
        exitDialog();
    }//GEN-LAST:event_closeDialog

    private void searchSimilarCards(String searchString, String sideString) {
        if (searchString.length() > 2) {
            List<Card> similarCards =
                    searchEngine.search(searchString, sideString);
            Batch similarCardsBatch = new Batch(similarCards);
            similarCardsListModel.setBatch(similarCardsBatch);
        } else {
            similarCardsListModel.clear();
        }
    }

    private void exitDialog() {
        setVisible(false);
        dispose();
    }

    boolean isKeepOpenSelected() {
        return keepOpenCheckBox.isSelected();
    }

    private class MyDocumentListener implements DocumentListener {

        private Card.Element cardSide;

        public MyDocumentListener(Card.Element cardSide) {
            this.cardSide = cardSide;
        }

        // pass through functions
        public void changedUpdate(DocumentEvent event) {
            documentChanged();
        }

        public void insertUpdate(DocumentEvent event) {
            documentChanged();
        }

        public void removeUpdate(DocumentEvent event) {
            documentChanged();
        }

        private void documentChanged() {
            if (showSimilarCards) {
                if (cardSide == Card.Element.FRONT_SIDE) {
                    searchSimilarCards(editCardPanel.getFrontSide(),
                            SearchEngine.FRONT_SIDE);
                } else {
                    searchSimilarCards(editCardPanel.getReverseSide(),
                            SearchEngine.REVERSE_SIDE);
                }
            }
        }
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JScrollPane batchListScrollPane;
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton cancelButton;
    private pauker.program.gui.swing.EditCardPanel editCardPanel;
    private javax.swing.JCheckBox keepOpenCheckBox;
    private javax.swing.JButton okButton;
    private pauker.program.gui.swing.UpdateLayoutList similarCardsList;
    private javax.swing.JPanel similarCardsPanel;
    private javax.swing.JLabel similarLabel;
    private javax.swing.JSeparator similarSeparator1;
    private javax.swing.JSeparator similarSeparator2;
    private javax.swing.JSplitPane splitPane;
    // End of variables declaration//GEN-END:variables
}
