# Copyright 2016 Open Source Robotics Foundation, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import time


class ExecutionEvent(object):

    """Structure for events generated by the Executor.

    Events can be jobs starting/finishing, commands starting/failing/finishing,
    commands producing output (each line is an event), or when the executor
    quits or failes.
    """

    # TODO: Make this a map of ID -> fields
    EVENT_IDS = [
        'JOB_STATUS',  # A report of running job states
        'QUEUED_JOB',  # A job has been queued to be executed
        'STARTED_JOB',  # A job has started to be executed
        'FINISHED_JOB',  # A job has finished executing (succeeded or failed)
        'ABANDONED_JOB',  # A job has been abandoned for some reason
        'STARTED_STAGE',  # A job stage has started to be executed
        'FINISHED_STAGE',  # A job stage has finished executing (succeeded or failed)
        'STAGE_PROGRESS',  # A job stage has executed partially
        'STDOUT',  # A status message from a job
        'STDERR',  # A warning or error message from a job
        'SUBPROCESS',  # A subprocess has been created
        'MESSAGE'
    ]

    def __init__(self, event_id, **kwargs):
        """Create a new event.

        :param event_id: One of the valid EVENT_IDS
        :param **kwargs: The additional data to be passed along with this event.
        """
        # Store the time this event was generated
        self.time = time.time()

        # Make sure the event ID is valid
        if event_id not in ExecutionEvent.EVENT_IDS:
            print(ExecutionEvent.EVENT_IDS)
            raise ValueError("The event ID %s is not a valid executor event." % event_id)

        # Store the event data
        self.event_id = event_id
        self.data = kwargs
