
! Copyright (C) 2002-2005 J. K. Dewhurst, S. Sharma and C. Ambrosch-Draxl.
! This file is distributed under the terms of the GNU Lesser General Public
! License. See the file COPYING for license details.

!BOP
! !ROUTINE: zfmtinp
! !INTERFACE:
complex(8) function zfmtinp(tsh,lmmax,nr,r,ld,zfmt1,zfmt2)
! !INPUT/OUTPUT PARAMETERS:
!   tsh   : .true. if the functions are in spherical harmonics (in,logical)
!   lmmax : number of (l,m) pairs, equal to (lmax+1)^2 (in,integer)
!   nr    : number of radial mesh points (in,integer)
!   r     : radial mesh (in,real(nr))
!   ld    : leading dimension (in,integer)
!   zfmt1 : first complex muffin-tin function in spherical harmonics/
!           coordinates (in,complex(ld,nr))
!   zfmt2 : second complex muffin-tin function in spherical harmonics/
!           coordinates (in,complex(ld,nr))
! !DESCRIPTION:
!   Calculates the inner product of two complex fuctions in the muffin-tin. In
!   other words, given two complex functions of the form
!   $$ f({\bf r})=\sum_{l=0}^{l_{\rm max}}\sum_{m=-l}^{l}f_{lm}(r)Y_{lm}
!    (\hat{\bf r}), $$
!   the function returns
!   $$ I=\sum_{l=0}^{l_{\rm max}}\sum_{m=-l}^{l}\int f_{lm}^{1*}(r)
!    f_{lm}^2(r)r^2\,dr\;. $$
!   Note that if {\tt tsh} is {\tt .false.} the functions are in spherical
!   coordinates rather than spherical harmonics. In this case $I$ is multiplied
!   by $4\pi/(l_{\rm max}+1)^2$.
!
! !REVISION HISTORY:
!   Created November 2003 (Sharma)
!EOP
!BOC
implicit none
! arguments
logical, intent(in) :: tsh
integer, intent(in) :: lmmax
integer, intent(in) :: nr
real(8), intent(in) :: r(nr)
integer, intent(in) :: ld
complex(8), intent(in) :: zfmt1(ld,nr),zfmt2(ld,nr)
! local variables
integer ir
real(8), parameter :: fourpi=12.566370614359172954d0
real(8) t1,t2
complex(8) z1
! automatic arrays
real(8) fr1(nr),fr2(nr),gr(nr)
! external functions
complex(8) zdotc
external zdotc
do ir=1,nr
  z1=zdotc(lmmax,zfmt1(:,ir),1,zfmt2(:,ir),1)*(r(ir)**2)
  fr1(ir)=dble(z1)
  fr2(ir)=aimag(z1)
end do
call fderiv(-1,nr,r,fr1,gr)
t1=gr(nr)
call fderiv(-1,nr,r,fr2,gr)
t2=gr(nr)
zfmtinp=cmplx(t1,t2,8)
if (.not.tsh) zfmtinp=zfmtinp*fourpi/dble(lmmax)
return
end function
!EOC


